/*----------------------------------------------------------------------------*/
/* Copyright 2015-2022 NXP                                                    */
/*                                                                            */
/* NXP Confidential. This software is owned or controlled by NXP and may only */
/* be used strictly in accordance with the applicable license terms.          */
/* By expressly accepting such terms or by downloading, installing,           */
/* activating and/or otherwise using the software, you are agreeing that you  */
/* have read, and that you agree to comply with and are bound by, such        */
/* license terms. If you do not agree to be bound by the applicable license   */
/* terms, then you may not retain, install, activate or otherwise use the     */
/* software.                                                                  */
/*----------------------------------------------------------------------------*/

/** @file
 *
 * Prototype of CLIF RF HAL API Event Mechanism with RTOS.
 *
 * Project:  PN7462AU
 *
 * $Date:
 * $Author:
 * $Revision:
 */

/* *****************************************************************************************************************
 * Includes
 * ***************************************************************************************************************** */
#include "ph_Datatypes.h"
#include "ph_NxpBuild.h"

#ifdef NXPBUILD__PHHAL_RF

#include "phhalRf.h"
#include "phhalSysSer.h"
#include "phhalRf_Event.h"
#include "phOsal.h"

/* *****************************************************************************************************************
 * Internal Definitions
 * ***************************************************************************************************************** */

/* *****************************************************************************************************************
 * Type Definitions
 * ***************************************************************************************************************** */

/* *****************************************************************************************************************
 * Global and Static Variables
 * Total Size: NNNbytes
 * ***************************************************************************************************************** */
/**
 * Global Status of ISR Event Register Status to Receive Events from ISR
 */
PH_NOINIT static phOsal_EventObj_t      HwEventObj;                               /**< Event Object */

/**
 * Source of Events
 */
PH_NOINIT static volatile phhalRf_EventSource_t gphhalRf_EvtLastSource;
/**
 * Destination of Events
 */
PH_NOINIT static volatile phhalRf_EventDestination_t gphhalRf_EvtLastDest;
/**
 * Event Detail
 */
PH_NOINIT static volatile phhalRf_EventDetail_t gphhalRf_EvtDetail;

/* *****************************************************************************************************************
 * Private Functions Prototypes
 * ***************************************************************************************************************** */

/* *****************************************************************************************************************
 * Public Functions
 * ***************************************************************************************************************** */
phStatus_t phhalRf_Event_Init(void)
{

    phOsal_EventCreate( &(HwEventObj.EventHandle), &HwEventObj );

    gphhalRf_EvtLastSource = E_PH_HALRF_EVT_SRC_NONE;
    gphhalRf_EvtLastDest = E_PH_HALRF_EVT_DEST_NONE;
    gphhalRf_EvtDetail = E_PH_HALRF_EVT_DETAIL_SUCCESS;
    return PH_ERR_SUCCESS;
}

phStatus_t phhalRf_Event_GetEventHandle(void *pHandle)
{

    if ( pHandle != NULL) {
        (* (phOsal_EventObj_t *)pHandle) = HwEventObj;
    }
    return PH_ERR_SUCCESS;
}

void phhalRf_Event_Post(phhalRf_EventType_t eEvtType, phhalRf_EventSource_t eSrc, phhalRf_EventDestination_t eDest)
{

        phOsal_EventPost( &(HwEventObj.EventHandle),E_OS_EVENT_OPT_NONE, (uint32_t)eEvtType, NULL );

        /* Set the Last Source */
        gphhalRf_EvtLastSource = eSrc;

        /* Set the Last Destination */
        gphhalRf_EvtLastDest = eDest;

        /* Set Default Event Detail */
        gphhalRf_EvtDetail = E_PH_HALRF_EVT_DETAIL_SUCCESS;

}

void phhalRf_Event_Consume(phhalRf_EventType_t eEvtType, phhalRf_EventSource_t eSrc)
{
    uint32_t xRet = 0;

    xRet = (uint32_t) phOsal_EventClear( &(HwEventObj.EventHandle), E_OS_EVENT_OPT_NONE, (uint32_t)eEvtType, NULL );

    if ((((uint32_t) xRet) & ((uint32_t) eEvtType)) == 0)
    {
        gphhalRf_EvtDetail = E_PH_HALRF_EVT_DETAIL_ERROR;
    }

    /* Set the Last Source */
    gphhalRf_EvtLastSource = eSrc;

}

phStatus_t phhalRf_Event_WaitAny(phhalRf_EventType_t eEvtType, uint32_t dwTimeoutCount,
    phhalRf_EventType_t *pRcvdEvt)
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint32_t xbits = 0;

    if (HwEventObj.EventHandle == NULL) /* In case it is not initialized yet */
            PH_RETURN_IF_TRUE_WITH_ERROR((eEvtType > E_PH_HALRF_EVT_TYPE_ALL), INVALID_PARAMETER, RF)

    /* Only Possible from HAL */

    /* Don't Clear flags after ISR and Don't Wait for All bits to be Set */
    phOsal_EventPend( &(HwEventObj.EventHandle), E_OS_EVENT_OPT_NONE, dwTimeoutCount, (uint32_t)eEvtType, (phOsal_EventBits_t *)&xbits );
    if ((xbits & ((uint32_t) eEvtType)) != 0)
    {
        status = PH_ERR_SUCCESS;
    }
    else
    {
        status = PH_ERR(IO_TIMEOUT, RF);
    }

    /* Return the Events That were Set */
    *pRcvdEvt = (phhalRf_EventType_t) xbits;
    return status;
}

phStatus_t phhalRf_Event_WaitAll(phhalRf_EventType_t eEvtType, uint32_t dwTimeoutCount,
    phhalRf_EventType_t *pRcvdEvt)
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint32_t xbits = 0;

    /* Only Possible from HAL */

    /* Don't Clear flags after ISR and Wait for All bits to be Set */
    phOsal_EventPend( &(HwEventObj.EventHandle), E_OS_EVENT_OPT_NONE, dwTimeoutCount, (uint32_t)eEvtType, (phOsal_EventBits_t *)&xbits );
    if ((xbits & ((uint32_t) eEvtType)) == ((uint32_t) eEvtType))
    {
        status = PH_ERR_SUCCESS;
    }
    else
    {
        status = PH_ERR(IO_TIMEOUT, RF);
    }

    /* Return the Events That were Set */
    *pRcvdEvt = (phhalRf_EventType_t) xbits;
    return status;
}

void phhalRf_Event_SetDetail(phhalRf_EventDetail_t eEvtDetail)
{
    gphhalRf_EvtDetail = eEvtDetail;
}

void phhalRf_Event_GetDetail(phhalRf_EventDetail_t * eEvent_Detail)
{
    *eEvent_Detail = gphhalRf_EvtDetail;
}

phStatus_t phhalRf_Event_DeInit(void)
{

    /* Delete the Event Group */
    phOsal_EventDelete( &(HwEventObj.EventHandle) );

    gphhalRf_EvtLastSource = E_PH_HALRF_EVT_SRC_NONE;
    gphhalRf_EvtLastDest = E_PH_HALRF_EVT_DEST_NONE;
    gphhalRf_EvtDetail = E_PH_HALRF_EVT_DETAIL_SUCCESS;
    return PH_ERR_SUCCESS;
}

void phhalRf_Event_PostwithDetail(phhalRf_EventType_t eEvtType, phhalRf_EventSource_t eSrc,
    phhalRf_EventDestination_t eDest, phhalRf_EventDetail_t eDetail)
{
    phhalRf_Event_Post(eEvtType, eSrc, eDest);
    phhalRf_Event_SetDetail(eDetail);
}

/* *****************************************************************************************************************
 * Private Functions
 * ***************************************************************************************************************** */

#endif /* NXPBUILD__PHHAL_RF */
