/*----------------------------------------------------------------------------*/
/* Copyright 2014, 2015, 2022, 2023 NXP                                       */
/*                                                                            */
/* NXP Confidential. This software is owned or controlled by NXP and may only */
/* be used strictly in accordance with the applicable license terms.          */
/* By expressly accepting such terms or by downloading, installing,           */
/* activating and/or otherwise using the software, you are agreeing that you  */
/* have read, and that you agree to comply with and are bound by, such        */
/* license terms. If you do not agree to be bound by the applicable license   */
/* terms, then you may not retain, install, activate or otherwise use the     */
/* software.                                                                  */
/*----------------------------------------------------------------------------*/

/** @file
 *
 * This file have the definition of functions, macros and enums which are internally used by RF hal.
 *
 * Project:  PN7462AU
 *
 * $Date: 2016-11-24 12:59:45 +0530 (Thu, 24 Nov 2016) $
 * $Author: Anish Ahammed (nxp86397) $
 * $Revision: 18760 $ (v04.18.00)
 */

#ifndef PHHALRF_INT_H
#define PHHALRF_INT_H

/* *****************************************************************************************************************
 * Includes
 * ***************************************************************************************************************** */
#include "ph_Datatypes.h"
#include "ph_NxpBuild.h"

#ifdef NXPBUILD__PHHAL_RF

#include "phSysSv_Data.h"
#include "phhalPmu.h"
#include "phhalRf.h"
#include "phhalRf_Event.h"
/* *****************************************************************************************************************
 * MACROS/Defines
 * ***************************************************************************************************************** */
PH_NOINIT extern phhalRf_Config_t gphhalRf_ActTargetSup;
#define PHHALRF_SET_ACTIVETARGET(X)                     gphhalRf_ActTargetSup = (phhalRf_Config_t)(X)
#define PHHALRF_GET_ACTIVETARGET()                      gphhalRf_ActTargetSup

/**
 * Hal Rx Buffer.
 */
#define PHHALRF_HAL_RXBUFFER_SIZE               260                                 /**< Hal Rx buffer size. */

PH_ALIGN(4) extern uint8_t gphhalRf_HalRxBuffer[PHHALRF_HAL_RXBUFFER_SIZE];
/* BL warnings */
#define PHHALRF_GET_HAL_RXBUFFER_SIZE()                     PHHALRF_HAL_RXBUFFER_SIZE   /**< Get Hal Rx buffer. */
#define PHHALRF_GET_HAL_RXBUFFER_STARTADDR()                &gphhalRf_HalRxBuffer[0]    /**< Get Hal Rx buffer address*/

PH_NOINIT extern uint16_t gphhalRf_RxMaxLength;
#define PHHALRF_SET_RX_MAX_LENGTH(X)                              (gphhalRf_RxMaxLength = (X))
#define PHHALRF_GET_RX_MAX_LENGTH()                               (gphhalRf_RxMaxLength)

PH_NOINIT extern uint16_t gphhalRf_AutoCollRunning;
#define PHHALRF_SET_AUTOCOLLRUNNING(X)                              (gphhalRf_AutoCollRunning = (X))
#define PHHALRF_GET_AUTOCOLLRUNNING()                               (gphhalRf_AutoCollRunning)

PH_NOINIT extern uint32_t gphhalRf_OpeMode;
#define PHHALRF_SET_OPE_MODE(X)                              (gphhalRf_OpeMode = (X))
#define PHHALRF_GET_OPE_MODE()                               (gphhalRf_OpeMode)

PH_NOINIT extern uint32_t gphhalRf_CE_A_State;
#define PHHAL_RF_SET_CE_A_STATE(X)                      (gphhalRf_CE_A_State = (X))
#define PHHAL_RF_GET_CE_A_STATE()                       gphhalRf_CE_A_State

#ifdef NXPBUILD__PHHAL_RF_EMD_2_5
PH_NOINIT extern phhalRf_Config_t gphhalRf_EMD;
#define PHHAL_RF_SET_EMD(X)                      (gphhalRf_EMD = (X))
#define PHHAL_RF_GET_EMD()                       gphhalRf_EMD
#endif /* NXPBUILD__PHHAL_RF_EMD_2_5 */

/**
 * Poll Guard Timer Enable
 */
PH_NOINIT extern phhalRf_Config_t gphhalRf_PollGuardTime_Flag;
#define PHHAL_RF_SET_POLL_GUARD_TIME_FLAG(X)                      (gphhalRf_PollGuardTime_Flag = (X))
#define PHHAL_RF_GET_POLL_GUARD_TIME_FLAG()                       gphhalRf_PollGuardTime_Flag
/**
 * FDT Timing Value
 */
PH_NOINIT extern uint32_t gphhalRf_FDT_Value;
#define PHHAL_RF_SET_FDT_VALUE(X)                      (gphhalRf_FDT_Value = (X))
#define PHHAL_RF_GET_FDT_VALUE()                       gphhalRf_FDT_Value

/**
 * FDT Timeing PreScalar
 */
PH_NOINIT extern uint32_t gphhalRf_FDT_Prescalar;
#define PHHAL_RF_SET_FDT_PRESCALAR(X)                      (gphhalRf_FDT_Prescalar = (X))
#define PHHAL_RF_GET_FDT_PRESCALAR()                       gphhalRf_FDT_Prescalar

#define PHHAL_RF_TXLDO_START_WAIT_ENABLE                    true
#define PHHAL_RF_TXLDO_START_WAIT_DISABLE                   (!PHHAL_RF_TXLDO_START_WAIT_ENABLE)

#define PHHAL_RF_TXLDO_START_WAIT_TIME_US                   300

#define PHHALRF_GET_RXBUFFER_ADDR()                     gpphhalRf_RxBuffer
#define PHHALRF_GET_RXBUFFER_SIZE()                     gphhalRf_RxBufferSize

#define PHHALRF_SET_RXBUFFER_ADDR(X)                     gpphhalRf_RxBuffer = (uint8_t *)(X)
#define PHHALRF_SET_RXBUFFER_SIZE(X)                     gphhalRf_RxBufferSize = (X)

/**
 * RFCA Configuration
 */
PH_NOINIT extern uint8_t gphhalRf_RFCA_Flag;
#define PHHAL_RF_SET_RFCA_FLAG(X)                      (gphhalRf_RFCA_Flag = (X))
#define PHHAL_RF_GET_RFCA_FLAG()                       gphhalRf_RFCA_Flag

/**
 * Maximum Transmit Buffer Length Possible 260 Bytes
 */
#define PHHAL_RF_TX_LENGTH_MAX (0x00000104UL)

#define PHHALRF_HAL_RX_MIN_LENGTH                                       4
#define PHHALRF_HAL_RX_MAX_LENGTH                                       508
#define PHHALRF_HAL_MIN_LPCD_ON_DURATION                                40U
#define PHHALRF_HAL_MAX_LPCD_ON_DURATION                                150U

/**
 * @name Rx multiple Error mask.
 * @{
 */
#define PHHAL_RF_RXMULTIPLE_LEN_MASK            0x0000001F  /**< Rx multiple Length Mask.*/
#define PHHAL_RF_RXMULTIPLE_CLERR_MASK          0x00000100  /**< Rx multiple error.*/
#define PHHAL_RF_RXMULTIPLE_DATAERR_MASK        0x00000200  /**< Rx multiple Data error. */
#define PHHAL_RF_RXMULTIPLE_PROERR_MASK         0x00000400  /**< Rx multiple protocol error.*/
#define PHHAL_RF_RXMULTIPLE_COLLDETT_MASK       0x00000800  /**< Rx Multiple collision error.*/
#define PHHAL_RF_RXMULTIPLE_LENERR_MASK         0x00001000  /**< Rx multiple length error.*/
#define PHHAL_RF_RXMULTIPLE_DATA_LEN            32          /**< Length of the whole single buffer.*/
/** @} */

#define PHHALRF_P2P_106_SYNC_BYTE                           0xF0    /**< Atr Req Sync Byte for Type A. */
#define PHHALRF_P2P_106_SYNC_BYTE_POS                       0x00    /**< Sync Byte Position in 106 kbps*/

#define PHHALRF_ATR_REQ_106_LEN_POS                         0x01    /**< Atr Req Length Position for 106 kbps*/
#define PHHALRF_ATR_REQ_212_424_LEN_POS                     0x00    /**< Atr Req Len Position for 212/424kbps*/

#define PHHALRF_P2P_CMD0_BYTE                               0xD4    /**< 18092 CMD0 byte. */

#define PHHALRF_ATR_REQ_CMD1_BYTE                           0x00    /**< Atr Req CMD1 byte. */
#define PHHALRF_WUP_REQ_CMD1_BYTE                           0x01    /**< WUP Req CMD1 byte. */

#define PHHALRF_P2P_106_CMD0_POS                            0x02    /**< Atr Req CMD0 byte Position for 106 kbps. */
#define PHHALRF_P2P_212_424_CMD0_POS                        0x01    /**< Atr Req CMD0 byte Position for 212/424 kbps. */

#define PHHALRF_P2P_106_CMD1_POS                            0x03    /**< Atr Req CMD1 byte Position for 106 kbps. */
#define PHHALRF_P2P_212_424_CMD1_POS                        0x02    /**< Atr Req CMD1 byte Position for 212/424 kbps. */

/**
 * @name Transceive State Machine Command
 * @{
 */
#define PHHAL_RF_CONFIG_TXV_IDLE_CMD            (0)       /**< Idle Command. */
#define PHHAL_RF_CONFIG_TXV_TRANSMIT_CMD        (1)       /**< Transmit Command. */
#define PHHAL_RF_CONFIG_TXV_RECEIVE_CMD         (2)       /**< Receive Command. */
#define PHHAL_RF_CONFIG_TXV_TRANSCEIVE_CMD      (3)       /**< Transceive command. */
#define PHHAL_RF_CONFIG_TXV_KEEP_CMD            (4)       /**< Keep Command. */
#define PHHAL_RF_CONFIG_TXV_LOOPBACK_CMD        (5)       /**< Loop Back Command. */

/* @}*/

/**
 * @name Transceive State Machine Triggers
 * @{
 */
#define PHHAL_RF_CONFIG_TXV_TRIGGER_STATE_IDLE      (1 << 0)    /**< Idle State. */
#define PHHAL_RF_CONFIG_TXV_TRIGGER_STATE_WAITTX    (1 << 1)    /**< Wait Transmit State. */
#define PHHAL_RF_CONFIG_TXV_TRIGGER_STATE_TX        (1 << 2)    /**< Transmit state.*/
#define PHHAL_RF_CONFIG_TXV_TRIGGER_STATE_WAITRX    (1 << 3)    /**< Wait for Receive.*/
#define PHHAL_RF_CONFIG_TXV_TRIGGER_STATE_WAITDATA  (1 << 4)    /**< Wait for Data State. */
#define PHHAL_RF_CONFIG_TXV_TRIGGER_STATE_RX        (1 << 5)    /**< Receiving State. */
/* @} */

/**
 * @name RF Active Error Codes
 * @{
 */
#define PHHAL_RF_CONFIG_ACTERR_NONE                (0)      /**< No RD Active Error. */
#define PHHAL_RF_CONFIG_ACTERR_EXTFLD_TIDT_ERROR   (1)      /**< External Field Detected within TIDT.*/
#define PHHAL_RF_CONFIG_ACTERR_EXTFLD_TADT_ERROR   (2)      /**< External Field Detected within TADT.*/
#define PHHAL_RF_CONFIG_ACTERR_EXTFLD_NDET_ERROR   (3)      /**< No External Filed Detected within TADT.*/
#define PHHAL_RF_CONFIG_ACTERR_PEER_FLD_ERROR      (4)      /**< Peer switch off RF Field without Transmitting.*/
/* @} */

#define PHHAL_RF_TIMER_COUNT_CAL_32BIT_LIMIT    2 << 22
#define PHHAL_RF_TIMER_VALUE_CAL_32BIT_LIMIT    2 << 18
#define PHHAL_RF_TIMER_VALUE_CAL_32BIT_LIMIT1   2 << 24
/**
 * for Tx/Rx Wait   : Prescale_Val = 0x7F + 1 = 128
 * for Timers       : Prescale_Val = 2^(1 + 6) = 128
 * Count = Time(us) * 13.56(MHz)
 * Count = Count / Prescale_Val
 */
#define PHHALRF_CAL_TIMER_COUNT(DWVALUE)                \
    do {                                                \
        if ((DWVALUE) > PHHAL_RF_TIMER_COUNT_CAL_32BIT_LIMIT)                          \
        {                                               \
            (DWVALUE) >>= 7;                            \
            (DWVALUE) *= 339;                           \
            (DWVALUE) /= 25;                            \
        }                                               \
        else                                            \
        {                                               \
            (DWVALUE) *= 339;                           \
            (DWVALUE) /= 25;                            \
            (DWVALUE) >>= 7 ;                           \
        }                                               \
    }while(0)

/**
 * Count = Time(us) * 20(MHz)
 */
#define PHHALRF_CAL_TIMER3_COUNT(DWVALUE)                \
    do {                                                 \
        (DWVALUE) *= 20;                                 \
    }while(0)

/**
 * for Tx/Rx Wait   : Prescale_Val = 0x7F + 1 = 128
 * for Timers       : Prescale_Val = 2^(1 + 6) = 128
 * Count = Count * Prescale_Val
 * Time(us) = Count / 13.56(MHz)
 */
#define PHHALRF_CAL_TIMER_VALUE(DWVALUE)                \
    do {                                                \
        if ((DWVALUE) < PHHAL_RF_TIMER_VALUE_CAL_32BIT_LIMIT)                           \
        {                                               \
            (DWVALUE) <<= 7;                            \
            (DWVALUE) *= 25;                            \
            (DWVALUE) /= 339;                           \
        }    \
        else if ((DWVALUE) < PHHAL_RF_TIMER_VALUE_CAL_32BIT_LIMIT1)     \
        {                                               \
            (DWVALUE) <<= 7;                            \
            (DWVALUE) /= 339;                           \
            (DWVALUE) *= 25;                            \
        }                                               \
        else                                            \
        {                                               \
            (DWVALUE) /= 339;                           \
            (DWVALUE) *= 25;                            \
            (DWVALUE) <<= 7;                            \
        }                                               \
    }while(0)

#define PHHALRF_TIMER_PRESCALAR_0x06            0x07            /**< Prescale used for Timers. */
#define PHHALRF_TIMER_MIN_VALUE_US              10              /**< Minimum Time timer can run(us). */
#define PHHALRF_TIMER_MAX_VALUE_US              9500000         /**< Maximum Time timer can run(us). */

#define PHHALRF_T1T_TXWAIT_BETWEEN_FRAMES       500

#define PHHALRF_MAX_EXCHANGE_TIME_OFFSET        100
#define PHHALRF_MAX_EXCHANGE_TIME               PHHAL_RF_CONFIG_PCDEXCHG_RXEVENT_WAIT - PHHALRF_MAX_EXCHANGE_TIME_OFFSET

#define PHHALRF_SET_ACTIVE_MODE()               \
    PH_REG_SET_BIT(CLIF_RF_CONTROL_REG, TX_AUTO_RFON);   \
    PH_REG_SET_BIT(CLIF_RF_CONTROL_REG, TX_AUTO_RFOFF);

#define PHHALRF_CLEAR_ACTIVE_MODE()               \
    PH_REG_CLEAR_BIT(CLIF_RF_CONTROL_REG, TX_AUTO_RFON);   \
    PH_REG_CLEAR_BIT(CLIF_RF_CONTROL_REG, TX_AUTO_RFOFF);

#define PHHALRF_GET_ACTIVE_MODE()               \
    ((PH_REG_TEST_BIT(CLIF_RF_CONTROL_REG, TX_AUTO_RFON))   \
        && (PH_REG_TEST_BIT(CLIF_RF_CONTROL_REG, TX_AUTO_RFOFF)))

#define PHHALRF_GET_RX_MULTIPLE()               \
    (PH_REG_TEST_BIT(CLIF_TRANSCEIVE_CONTROL_REG, RX_MULTIPLE_ENABLE))

#define PHHALRF_GET_INITAITOR_MODE()               \
    (PH_REG_TEST_BIT(CLIF_TRANSCEIVE_CONTROL_REG, INITIATOR))

#define PHHALRF_GET_JEWEL_MODE()               \
    (PH_REG_TEST_BIT(CLIF_TRANSCEIVE_CONTROL_REG, TX_FRAMESTEP_ENABLE))

/**
 * Timers
 */
/* Initiator */
#define PHHALRF_FIELD_RESET_TIMER                       E_PH_HALRF_TMR_TYPE_TIMER0
#define PHHALRF_POLL_GUARD_TIMER_US                     E_PH_HALRF_TMR_TYPE_TIMER0
#define PHHALRF_FDT_TIMER                               E_PH_HALRF_TMR_TYPE_TIMER1
#define PHHALRF_EXT_RF_ON_INI_TIMER                     E_PH_HALRF_TMR_TYPE_TIMER2
#define PHHALRF_TCOCP_TIMER                             E_PH_HALRF_TMR_TYPE_TIMER3

/* Target */
#define PHHALRF_APC_TIMER                               E_PH_HALRF_TMR_TYPE_TIMER0
#define PHHALRF_FELICA_SC3372_TIMER                     E_PH_HALRF_TMR_TYPE_TIMER1
#define PHHALRF_EXT_RF_ON_TAR_TIMER                     E_PH_HALRF_TMR_TYPE_TIMER2
#define PHHALRF_PBF_ALM_FW_RESET_TIMER                  E_PH_HALRF_TMR_TYPE_TIMER3

#define PHHALRF_GET_P2P_18092_212_424()                                                                             \
    (                                                                                                               \
        (                                                                                                           \
            ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_212_424_CMD0_POS)))                           \
                == PHHALRF_P2P_CMD0_BYTE)                                                                           \
        )                                                                                                           \
    )

#define PHHALRF_GET_ATR_212_424()                                                                                   \
    (                                                                                                               \
        (                                                                                                           \
            ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_212_424_CMD0_POS)))                           \
                == PHHALRF_P2P_CMD0_BYTE)                                                                           \
                &&                                                                                                  \
                (                                                                                                   \
                    ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_212_424_CMD1_POS)))                   \
                        == PHHALRF_ATR_REQ_CMD1_BYTE)                                                               \
                )                                                                                                   \
        )                                                                                                           \
    )

#define PHHALRF_GET_WUP_212_424()                                                                                   \
    (                                                                                                               \
        (                                                                                                           \
            ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_212_424_CMD0_POS)))                           \
                == PHHALRF_P2P_CMD0_BYTE)                                                                           \
                &&                                                                                                  \
                (                                                                                                   \
                    ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_212_424_CMD1_POS)))                   \
                        == PHHALRF_WUP_REQ_CMD1_BYTE)                                                               \
                )                                                                                                   \
        )                                                                                                           \
    )

#define PHHALRF_GET_ATR_WUP_212_424()                                                                               \
    (                                                                                                               \
        (                                                                                                           \
            ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_212_424_CMD0_POS)))                           \
                == PHHALRF_P2P_CMD0_BYTE)                                                                           \
                &&                                                                                                  \
                (                                                                                                   \
                    ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_212_424_CMD1_POS)))                   \
                        == PHHALRF_ATR_REQ_CMD1_BYTE)                                                               \
                        ||                                                                                          \
                        ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_212_424_CMD1_POS)))               \
                            == PHHALRF_WUP_REQ_CMD1_BYTE)                                                           \
                )                                                                                                   \
        )                                                                                                           \
    )

#define PHHALRF_GET_P2P_18092_106()                                                                                 \
    (                                                                                                               \
        (                                                                                                           \
            ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_SYNC_BYTE_POS)))                          \
                == PHHALRF_P2P_106_SYNC_BYTE)                                                                       \
                &&                                                                                                  \
                ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_CMD0_POS))) == PHHALRF_P2P_CMD0_BYTE) \
        )                                                                                                           \
    )

#define PHHALRF_GET_ATR_106()                                                                                       \
    (                                                                                                               \
        (                                                                                                           \
            ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_SYNC_BYTE_POS)))                          \
                == PHHALRF_P2P_106_SYNC_BYTE)                                                                       \
                &&                                                                                                  \
                ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_CMD0_POS))) == PHHALRF_P2P_CMD0_BYTE) \
                &&                                                                                                  \
                (                                                                                                   \
                    ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_CMD1_POS)))                       \
                        == PHHALRF_ATR_REQ_CMD1_BYTE)                                                               \
                )                                                                                                   \
        )                                                                                                           \
    )

#define PHHALRF_GET_WUP_106()                                                                                       \
    (                                                                                                               \
        (                                                                                                           \
            ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_SYNC_BYTE_POS)))                          \
                == PHHALRF_P2P_106_SYNC_BYTE)                                                                       \
                &&                                                                                                  \
                ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_CMD0_POS))) == PHHALRF_P2P_CMD0_BYTE) \
                &&                                                                                                  \
                (                                                                                                   \
                    ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_CMD1_POS)))                       \
                        == PHHALRF_WUP_REQ_CMD1_BYTE)                                                               \
                )                                                                                                   \
        )                                                                                                           \
    )

#define PHHALRF_GET_ATR_WUP_106()                                                                                   \
    (                                                                                                               \
        (                                                                                                           \
            ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_SYNC_BYTE_POS)))                          \
                == PHHALRF_P2P_106_SYNC_BYTE)                                                                       \
                &&                                                                                                  \
                ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_CMD0_POS))) == PHHALRF_P2P_CMD0_BYTE) \
                &&                                                                                                  \
                (                                                                                                   \
                    ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_CMD1_POS)))                       \
                        == PHHALRF_ATR_REQ_CMD1_BYTE)                                                               \
                        ||                                                                                          \
                        ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHALRF_P2P_106_CMD1_POS)))                   \
                            == PHHALRF_WUP_REQ_CMD1_BYTE)                                                           \
                )                                                                                                   \
        )                                                                                                           \
    )

#define PHHALRF_GET_HLTA()                                                                                          \
    ((*(((uint8_t *)PHHALRF_GET_RXBUFFER_ADDR()) + PHHALRF_HALTA_BYTE0_POS) == PHHALRF_HALTA_BYTE0)                 \
        && (*(((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR()) + PHHALRF_HALTA_BYTE1_POS)) == PHHALRF_HALTA_BYTE1))        \

#define PHHALRF_GET_REQA()                                                                                          \
    (*(((uint8_t *)PHHALRF_GET_RXBUFFER_ADDR()) + PHHALRF_REQA_BYTE1_POS) == PHHALRF_REQA_BYTE1)

#define PHHALRF_GET_WUPA()                                                                                          \
    (*(((uint8_t *)PHHALRF_GET_RXBUFFER_ADDR()) + PHHALRF_WUPA_BYTE1_POS) == PHHALRF_WUPA_BYTE1)

#define PHHALRF_GET_AC1()                                                                                           \
    (*(((uint8_t *)PHHALRF_GET_RXBUFFER_ADDR()) + PHHALRF_AC1_BYTE1_POS) == PHHALRF_AC1_BYTE1)

#define PHHALRF_GET_AC2()                                                                                           \
    (*(((uint8_t *)PHHALRF_GET_RXBUFFER_ADDR()) + PHHALRF_AC1_BYTE1_POS) == PHHALRF_AC2_BYTE1)

#define PHHALRF_GET_AC3()                                                                                           \
    (*(((uint8_t *)PHHALRF_GET_RXBUFFER_ADDR()) + PHHALRF_AC1_BYTE1_POS) == PHHALRF_AC3_BYTE1)

#define PHHALRF_GET_SENSF_REQ()                                                                                     \
    (((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHAL_RF_SENSF_REQ_LEN_POS))) == PHHAL_RF_SENSF_REQ_LEN)         \
        && ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHAL_RF_SENSF_REQ_BYTE1_POS))) == PHHAL_RF_SENSF_REQ_BYTE1))

#define PHHALRF_GET_SENSF_SC_DEFAULT()                                                                              \
    (((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHAL_RF_SENSF_REQ_SC_BYTE1_POS)))                               \
        == PHHAL_RF_SENSF_REQ_SC_DEFAULT_BYTE1)                                                                     \
        && ((*((uint8_t *)(PHHALRF_GET_RXBUFFER_ADDR() + PHHAL_RF_SENSF_REQ_SC_BYTE2_POS)))                         \
            == PHHAL_RF_SENSF_REQ_SC_DEFAULT_BYTE2))

#define PHHALRF_GET_SENSF_SC(X)                                                                                     \
    phUser_MemCmp((PHHALRF_GET_RXBUFFER_ADDR() + PHHAL_RF_SENSF_REQ_SC_BYTE1_POS), X, PHHAL_RF_SYSTEM_CODE_LEN)

/* #define NXPBUILD__PHHAL_RF_TIMER_AUTO */                     /**< Use the RF HAL Auto Prescalar functionality. */
/* #define NXPBUILD__PHHAL_RF_RESET_RF_TIMER */                 /**< Use the Field Reset Functionality with Timer. */
#define PHHAL_RF_AGC_UPDATE_SKIP                                        0xFFFFFFFF

#define PHHAL_RF_LOAD_RM_AGC                                                                                        \
    (PH_REG_GET(CLIF_AGC_CONFIG0_REG) |                                                                             \
(CLIF_AGC_CONFIG0_REG_AGC_INPUT_SEL_MASK | CLIF_AGC_CONFIG0_REG_AGC_MODE_SEL_MASK))

#define PHHAL_RF_LOAD_CM_AGC        \
    (PH_REG_GET(CLIF_AGC_CONFIG0_REG) &                                                                             \
    ((~CLIF_AGC_CONFIG0_REG_AGC_INPUT_SEL_MASK) | CLIF_AGC_CONFIG0_REG_AGC_MODE_SEL_MASK))

/* *****************************************************************************************************************
 * Types/Structure Declarations
 * ***************************************************************************************************************** */

/**
 * Mode Detected in Target mode.
 */
typedef enum phhalRf_ModeDetected{
    E_MODE_DETECTED_ISO14443A,
    E_MODE_DETECTED_ISO18092,
    E_MODE_DETECTED_FELICA,
    E_MODE_DETECTED_ISO14443B,
    E_MODE_DETECTED_MAX
}phhalRf_ModeDetected_t;

typedef enum phhalRf_RssiTrigger
{
    E_PHRF_RSSI_TRIGGER_RF_ON,
    E_PHRF_RSSI_TRIGGER_TX_DATA,
    E_PHRF_RSSI_TRIGGER_TIMER
} phhalRf_RssiTrigger_t;

/* Technologies to be Configured to Listen in Target mode
 * NEVER CHANGE THIS ORDER. */
typedef enum phhalRf_ListenTech_Configured{
    E_LISTENTECH_CONFIGURED_UNKNOWN = (0 << 0),
    E_ISO14443A = (1 << 0),
    E_FELICA = (1 << 1),
    E_TYPE_B = (1 << 2)
}phhalRf_ListenTech_Configured_t;

typedef enum phhalRf_AGCLoadConfig{
    E_PHRF_AGC_LOAD_INIT,
    E_PHRF_AGC_LOAD_BOOT,
    E_PHRF_AGC_LOAD_FIELDON,
    E_PHRF_AGC_LOAD_FIELDOFF,
    E_PHRF_AGC_LOAD_PROTOCOL
}phhalRf_AGCLoadConfig_t;

typedef struct phhalRf_AgcConfig
{
   uint32_t dwAgcConfig0;
   uint32_t dwAgcConfig1;
   uint32_t dwAgcInput;
}phhalRf_AgcConfig_t;

/* *****************************************************************************************************************
 * Extern Variables
 * ***************************************************************************************************************** */
extern uint8_t * gpphhalRf_RxBuffer;                    /**< Rx Buffer location.*/
extern uint16_t gphhalRf_RxBufferSize;                  /**< Rx Buffer size.*/
extern uint32_t gphhalRf_SLALM;                         /**< Variable to store the SL-ALM supported. */
extern pphhalRf_CallbackFunc_t gphhalRf_CallbackFunc;
extern uint32_t gphhalRf_RegIntrpts;

extern phhalRf_Config_t gphhalRf_NfcIp1;
#define PHHAL_RF_SET_NFCIP1(X)                          (gphhalRf_NfcIp1 = (X))
#define PHHAL_RF_GET_NFCIP1()                           (gphhalRf_NfcIp1)

/* *****************************************************************************************************************
 * Function Prototypes
 * ***************************************************************************************************************** */
/**
 * This function will Initialize all the global variables and buffers ussed by the Rf Hal.
 */
void phhalRf_ExgGlobalInit(void);

/**
 * This function will set the Initiator Base Addresses for Load Protocol.
 * This function have to be called before #phhalRf_LoadProtocol_Initiator
 *
 * @param[in] psBaseAddr Pointer to base addresses structure. \ref phhalRf_LP_BaseAddr_I_t
 * @return Status of the API
 * @retval #PH_ERR_SUCCESS  Operation successful.
 * @maskedret #PH_ERR_INVALID_PARAMETER Invalid Base Address Passed.
 */
phStatus_t phhalRf_LP_SetInitiatorBaseAddr(phSysSv_Clif_LP_BaseAddr_I_t * psBaseAddr);

/**
 * This function will set the Target Base Addresses for Load Protocol.
 * This function have to be called before #phhalRf_LoadProtocol_Target
 *
 * @param[in] psBaseAddr Pointer to base addresses structure. \ref phhalRf_LP_BaseAddr_I_t
 *
 * @return Status of the API
 * @retval #PH_ERR_SUCCESS  Operation successful.
 * @maskedret #PH_ERR_INVALID_PARAMETER Invalid Base Address Passed.
 */
phStatus_t phhalRf_LP_SetTargetBaseAddr(phSysSv_Clif_LP_BaseAddr_T_t * psBaseAddr);

/**
 * @brief This function gives the phSysSv_Clif_MF_Authenticate_Parameters_t structure required for MiFare
 *        Authentication
 *
 * @param[out] sMF_Auth_Param phSysSv_Clif_MF_Authenticate_Parameters_t structure
 */
void phhalRf_Get_MF_Auth_Param(phSysSv_Clif_MF_Authenticate_Parameters_t ** sMF_Auth_Param);

/**
 * @brief This function defines when to enable the Load Key for MiFare Authentication
 *
 * @param[out] dwMFC_Enable Parameter which tells to enable the Load Key for MiFare Authentication
 */
void phhalRf_Get_MFC_Enable(uint32_t ** dwMFC_Enable);

/**
 * @brief This function set up the Tx and Rx buffer.
 *
 * @param[in] pTxBuffer Transmit Buffer address.
 * @param[in] pRxBuffer Receive Buffer address.
 * @param[in] wRxMaxLength Rx Buffer maximum length.
 *
 * @return Status of the API
 * @retval #PH_ERR_SUCCESS  Operation successful.
 * @retval Other Depends on the underlying components.
 */
phStatus_t phhalRf_SetUpTxRxBuffer(const uint8_t * pTxBuffer, const uint8_t * pRxBuffer);

/**
 * This function does a Clean Up for Target mode.
 */
void phhalRf_TargetCleanUp(void);

/**
 * This function will enable the IRQs carefully.
 *
 * @param[in] dwEnableIrqs  IRQs to be enabled.
 * @param[in] dwClearIrqs   IRQs to be cleared.
 * @param[in] eConsumeEvents Events to be consumed.
 */
void phhalRf_EnableIrq(uint32_t dwEnableIrqs, uint32_t dwClearIrqs, phhalRf_EventType_t eConsumeEvents);

/**
 * This function will enable the IRQs carefully.
 *
 * @param[in] dwClearIrqs   IRQs to be cleared.
 * @param[in] eConsumeEvents Events to be consumed.
 */
void phhalRf_ClearIrq(uint32_t dwClearIrqs, phhalRf_EventType_t eConsumeEvents);

phhalRf_LP_I_Tx_t phhalRf_GetLoadedInitiatorTx(void);

phhalRf_LP_I_Rx_t phhalRf_GetLoadedInitiatorRx(void);

phhalRf_LP_T_Tx_t phhalRf_GetLoadedTargetTx(void);
phhalRf_LP_T_Rx_t phhalRf_GetLoadedTargetRx(void);
/**
 * Set the settings for HID
 */
void phhalRf_Set_HID(phhalRf_Config_t eConfig);

void phhalRf_MFC_ReAuth(void);

phStatus_t phhalRf_ConfigNfcOp(phhalRf_Config_t eConfig);

phStatus_t phhalRf_LP_Initiator(phhalRf_LP_I_Tx_t eTxNum, phhalRf_LP_I_Rx_t eRxNum);

phStatus_t phhalRf_LP_Target(phhalRf_LP_T_Tx_t eTxNum, phhalRf_LP_T_Rx_t eRxNum);

phStatus_t phhalRf_ResetInitiatorTx(void);
phStatus_t phhalRf_ResetInitiatorRx(void);

phStatus_t phhalRf_ResetTargetTx(void);
phStatus_t phhalRf_ResetTargetRx(void);

void phhalRf_ClearLPInitiator(void);
void phhalRf_ClearLPTarget(void);

void phhalRf_EnableRmMixer(phhalRf_Config_t bRMEnable);

/**
 * This function setup the minimum InterFrame Delay.
 *
 * @param[in] dwGuardTimeUs Guard Time(in us)
 * @param[in] eOption Option for Tx or Rx. \ref phhalRf_Tx_Rx_t
 *
 * @return Status of the API
 * @retval #PH_ERR_SUCCESS  Operation successful.
 * @retval Other Depends on the underlying components.
 *
 */
phStatus_t phhalRf_SetMinGuardTime(uint32_t dwGuardTimeUs, phhalRf_Tx_Rx_t eOption);

phStatus_t phhalRf_RxActiveRfOnHandle(void);

void phhalRf_AGCHandling(phhalRf_AGCLoadConfig_t dwEntry);

void phhalRf_RestoreAgc(phhalRf_AgcConfig_t * pAgcConfig);

void phhalRf_InitaitorGlobalInit(void);
void phhalRf_DisableRfDriver(void);
void phhalRf_EnableRfDriver(void);
#endif /* NXPBUILD__PHHAL_RF */
#endif /* PHHALRF_INT_H */
