/*
 *                    Copyright (c), NXP Semiconductors
 *
 *                       (C) NXP Semiconductors 2015
 *
 *         All rights are reserved. Reproduction in whole or in part is
 *        prohibited without the written consent of the copyright owner.
 *    NXP reserves the right to make changes without notice at any time.
 *   NXP makes no warranty, expressed, implied or statutory, including but
 *   not limited to any implied warranty of merchantability or fitness for any
 *  particular purpose, or that the use will not infringe any third party patent,
 *   copyright or trademark. NXP must not be liable for any loss or damage
 *                            arising from its use.
 */

/** @file
 *
 * phhalRf_OcProt.h :  This file contains the functionality for the Overcurrent Protection.
 *
 * Project:  PN640
 *
 * $Date: 2016-11-21 14:48:37 +0530 (Mon, 21 Nov 2016) $
 * $Author: Anish Ahammed (nxp86397) $
 * $Revision: 18753 $ (v04.18.00)
 */

#ifndef PHHALRF_OCPROT_H
#define PHHALRF_OCPROT_H

/* *****************************************************************************************************************
 * Includes
 * ***************************************************************************************************************** */
#include "ph_Datatypes.h"
#include "ph_NxpBuild.h"

#ifdef NXPBUILD__PHHAL_RF

#ifdef NXPBUILD__PHHAL_RF_DPC

#include "ph_Status.h"
#include "phCfg_EE.h"
#include "PN7462AU/PN7462AU_clif.h"
#include "ph_Reg.h"
/* *****************************************************************************************************************
 * MACROS/Defines
 * ***************************************************************************************************************** */
#define PHHAL_RF_OCPROT_NUMGEARS 15U
#define PHHAL_RF_OCPROT_MINGEAR  0U

#define PHHAL_RF_OCPROT_DEBUG_GPIO                                              8
#define PHHAL_RF_OCPROT_DEBUG_GPIO_SET                                          true
#define PHHAL_RF_OCPROT_DEBUG_GPIO_CLEAR                                        0
/*
 * LOOKUPTABLE VALUES CODING:
 * Value        TX_CW_GSN_RM    TX_CW_AMPLITUDE_RM      TX_CW_TO_MAX_RM         TX_CW_AMP_REF2TVDD
 * Bits:        7-4             3-2                     1                       0
 */

/* Macros for accessing the values in the fields */
#define OCPROT_CONTROL_ENABLED
#define PHHAL_RF_OCPROT_CONTROL_ENABLED_MASK                                 (0x01UL)
#define PHHAL_RF_OCPROT_CONTROL_ENABLED_POS                                  (0UL)

#define OCPROT_CONTROL_STARTGEAR
#define PHHAL_RF_OCPROT_CONTROL_STARTGEAR_MASK                               (0xF0UL)
#define PHHAL_RF_OCPROT_CONTROL_STARTGEAR_POS                                (4UL)

#define OCPROT_CONTROL_GEARSTEP
#define PHHAL_RF_OCPROT_CONTROL_GEARSTEP_MASK                                (0x0EUL)
#define PHHAL_RF_OCPROT_CONTROL_GEARSTEP_POS                                 (1UL)

#define OCPROT_STRATEGY_STRATEGY
#define PHHAL_RF_OCPROT_STRATEGY_STRATEGY_MASK                               (0x03UL)
#define PHHAL_RF_OCPROT_STRATEGY_STRATEGY_POS                                (0UL)

#define OCPROT_AGCFASTMODE_DURATION_FASTMODE_ENABLED
#define PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_STEPSIZE_ENABLED_POS            (13UL)
#define PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_STEPSIZE_ENABLED_MASK           \
    (1 << PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_STEPSIZE_ENABLED_POS)
#define PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_DURATION_ENABLED_POS            (10UL)
#define PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_DURATION_ENABLED_MASK           \
    (1 << PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_DURATION_ENABLED_POS)
#define PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_STEPSIZE_VALUE_POS              (11UL)
#define PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_STEPSIZE_VALUE_MASK             \
    (0x3 << PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_STEPSIZE_VALUE_POS)
#define PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_DURATION_VALUE_POS              (0UL)
#define PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_DURATION_VALUE_MASK             \
    (0x3FF << PHHAL_RF_OCPROT_AGCFASTMODE_CONTROL_DURATION_VALUE_POS)

#define UNUSED_VALUE(func)   (void)(func)  /**< The return value of the function is ignored. */

#define PHHAL_RF_TX_OCP_GETFIELD(value, field) ( ( (value) & PHHAL_RF_##field##_MASK ) >> (PHHAL_RF_##field##_POS) )

#define PHHAL_RF_OCPROT_PULSE_TEST_SIGNAL(a)                            \
    if ( gpkphCfg_EE_HW_RfDPC->bDebug )                                 \
        do {                                                            \
            uint8_t idx = (a);                                          \
            while (idx--)                                               \
            {                                                           \
                /* PH_HALREG_SETBITN(PCR_PADOUT_REG, SMU_PADOUT_REG_PADOUT_CLKREQ_POS);   */ \
                /* PH_HALREG_CLEARBITN(PCR_PADOUT_REG, SMU_PADOUT_REG_PADOUT_CLKREQ_POS); */ \
                PH_HAL_GPIO_SETGPIOVAL(PHHAL_RF_OCPROT_DEBUG_GPIO, PHHAL_RF_OCPROT_DEBUG_GPIO_SET); \
                PH_HAL_GPIO_SETGPIOVAL(PHHAL_RF_OCPROT_DEBUG_GPIO, PHHAL_RF_OCPROT_DEBUG_GPIO_CLEAR); \
            };                                                          \
        } while(0)

#define PHHAL_RF_DISABLE_TX_OCP_TIMER() \
    PH_REG_CLEAR_BIT(CLIF_TIMER3_CONFIG_REG, T3_ENABLE); /**< Timer 3 is used for the Tx Over Current Protection. */

#define PHHAL_RF_ENABLE_TX_OCP_TIMER()  PH_REG_SET_BIT(CLIF_TIMER3_CONFIG_REG, T3_ENABLE);

#define PHHAL_RF_OCPROT_GEAR_DEFAULT                               (0x00UL)
#define PHHAL_RF_OCPROT_GEAR_RESTORE                               (0x80UL)
#define PHHAL_RF_OCPROT_GEAR_MASK                                   (0x0FUL)

#define PHHAL_RF_OCPROT_PCD_SHAPE_CONFIG_SIZE_LUT_MASK             (0x0FUL)
#define PHHAL_RF_OCPROT_RXGAIN_HPCF_CONFIG_SIZE_LUT_MASK           (0xF0UL)
#define PHHAL_RF_OCPROT_RXGAIN_HPCF_CONFIG_SIZE_LUT_POS            (0x4UL)
#define PHHAL_RF_OCPROT_LOAD_DEPENDENT_RXGAIN_ENABLE_MASK          (0x80UL)
#define PHHAL_RF_OCPROT_LOAD_DEPENDENT_RXGAIN_ENABLE_POS           (0x7UL)
/* *****************************************************************************************************************
 * Types/Structure Declarations
 * ***************************************************************************************************************** */
typedef enum phhalRf_OcProt_Direction
{
    E_PH_HALRF_OCPROT_DIRECTION_DIRECT  = 0,
    E_PH_HALRF_OCPROT_DIRECTION_DOWN    = 1,
    E_PH_HALRF_OCPROT_DIRECTION_UP      = 2,
    E_PH_HALRF_OCPROT_DIRECTION_MAX     = 3
} phhalRf_OcProt_Direction_t;

typedef enum phhalRf_OcProt_FastMode
{
    E_PH_HALRF_OCPROT_FASTMODE_NOT_RUNNING    = 0,
    E_PH_HALRF_OCPROT_FASTMODE_STILL_RUNNING  = 1,
    E_PH_HALRF_OCPROT_FASTMODE_FINISHED       = 2
} phhalRf_OcProt_FastMode_t;

typedef PH_PACK_STRUCT_BEGIN struct phhalRf_DPCConfig_t
{
    /* Sets the value for the periodic regulation. Time base is 1/20Mhz. (Example: Value of 20000 is equal to 1ms) */
    uint16_t  wControlCycle;

    /* Controls AGC FastMode (StepSizeEnabled: 13 + StepSize: 12..11 + DurationEnabled: 10 + Duration: 9..0 ) */
    uint16_t  wAgcFastModeConfig;

    uint16_t  wAgcTrshLow; /* Low threashold for gearshift */

    /* Guard time after AGC fast mode has been triggered. This happens in the following scenarios:
    - End of Receive
    - End of Transmit
    - After a gear switch
    Time base is 1/20MHz (Example: Value of 2000 is equal to 100us) */
    uint16_t  wGuardTimeFastMode;

    /* Guard time after SoF or SC detection. This is to avoid any TxOCP regulation between SoF/SC
     * and actual begin of reception. Time base is 1/20MHz (Example: Value of 2000 is equal to 100us) */
    uint16_t  wGuardTimeSofDetected;

    /* Guard time after Gear Switch during FieldOn instruction.
     * Time base is 1/20MHz (Example: Value of 2000 is equal to 100us) */
    uint16_t  wGuardTimeFieldOn;
    uint16_t  wAgcTrshHigh[15]; /* High threasholds for each gear */
    uint8_t  bOcProtControl; /* Control byte (StartGear: 7..4 bits + GearStep: 3..1 bits + OcProtLoopEnabled: 0 bit ) */
    uint8_t  bAgcXi; /* Compensation value for the AGC */
    uint8_t  bDebug; /* Enable/Disable debug signals */
    uint8_t  bAgcShiftValue; /* Shift value for AGC dynamic low threashold adjustment */
    uint8_t  bSizeOfLUT; /* Number of fields in the following configuration look up table */
    uint8_t  bConfigLUT[15]; /* Look up table for configuration values */
} PH_PACK_STRUCT_END phhalRf_DPCConfig_t_t;

typedef PH_PACK_STRUCT_BEGIN struct phhalRf_PcdShapeConfig
{
    uint32_t  dwConfiguration[16]; /* The lookup table for the configuration for PCD shaping. */
    uint8_t  bSizeOfLUT; /* Number of elements in the following configuration look up table. */
} PH_PACK_STRUCT_END phhalRf_PcdShapeConfig_t;

/* *****************************************************************************************************************
 *   Extern Variables
 * ***************************************************************************************************************** */

/* *****************************************************************************************************************
 *   Function Prototypes
 * ***************************************************************************************************************** */

/**
 * Instructions used for the Overcurrent Protection Feature.
 * \note TODO.
 */
void phhalRf_OcProt_ControlLoop(void);

/**
 * Init Function for the Overcurrent Protection Feature.
 * \note TODO.
 */
void phhalRf_OcProt_Init(void);

void phhalRf_OcProt_DeInit(void);

/**
 * Resets the gear to the initial gear (prerequisite for RF on)
 * \note TODO.
 */
void phhalRf_OcProt_SetGear(uint8_t bGearLoc);

/**
 * Function to sign OcProt control algorithm of RF on event
 * \note TODO.
 */
void phhalRf_OcProt_FieldOn(void);

/**
 * Function to sign OcProt control algorithm of RF off event
 * \note TODO.
 */
void phhalRf_OcProt_FieldOff(void);

/**
 * Start the AGC fast mode, shall not be blocking! (Can be called from ISR context)
 * \note TODO.
 */
void phhalRf_OcProt_EnableFastMode(void);

/**
 * Set the guard timer which is used to prevent TxOCP from triggering AGC measurements.
 * \param[in] dwTimerValue Timer value to be used. Base is HFO (20MHz).
 */
void phhalRf_OcProt_SetGuardTimer(uint32_t dwTimerValue);

/**
 * TODO.
 * \returns status code
 * \retval PH_ERR_SUCCESS Gear switch occured.
 * \retval PH_ERR_INACTIVE No gear switch.
 */
phStatus_t phhalRf_OcProt_ShiftGear(phhalRf_OcProt_Direction_t eDirection, uint8_t bFixGear);

/**
 * Checks if adjusting of shaping configuration is required after gear switch
 *
 * This API is called either after a gear switch or in case of LoadProtocol. It performs a lookup in
 * EEPROM and checks if the new gear has assinged shaping configuration.
 * \param[in] TechnotAndBaudrate: Input parameter identifying protocol and baudrate.
 * \param[in] blIsInitial: TRUE in case API called via LoadProtocol to indicate to start from initial value
 *                          0... A-106, 1... A-212, 2... A-424, 3... A-848
 *                          4... B-106, 5... B-212, 6... B-424, 7... B-848
 *                          8... F-212, 9... F-424
 *                         10... 15693_ASK100, 11... 15693_ASK10
 *                         12... ISO18k3m3_TARI_18_88
 * \returns void.
 */
void phhalRf_OcProt_AdjustShaping(phhalRf_OcProt_Direction_t eDir);

void phhalRf_OcProt_AdjustRXGain(phhalRf_OcProt_Direction_t eDir);
/**
 * TODO.
 * \note TODO.
 * \return Status code
 * \retval PH_ERR_SUCCESS Gear switch occured.
 * \retval PH_ERR_INACTIVE No gear switch.
 */
phStatus_t phhalRf_OcProt_AGCUpdate(void);

/**
 * Getter function to retrive the current gear
 * \return void
 */
void phhalRf_OcProt_GetCurrentGear(uint8_t* bCurrentGear);

/**
 * Getter function to retrive the AGC low threshold for the current gear
 * \return void
 */
void phhalRf_OcProt_GetAGCLow(uint16_t* wDynamicAgcLow);

void phhalRf_OcProt_Precondition(void);
void phhalRf_OcProt_Precondition_RxGain(void);

#endif /* NXPBUILD__PHHAL_RF_DPC */

#endif /* NXPBUILD__PHHAL_RF */

#endif /* PHHALRF_OCPROT_H */
