/*----------------------------------------------------------------------------*/
/* Copyright 2020 NXP                                                         */
/*                                                                            */
/* NXP Confidential. This software is owned or controlled by NXP and may only */
/* be used strictly in accordance with the applicable license terms.          */
/* By expressly accepting such terms or by downloading, installing,           */
/* activating and/or otherwise using the software, you are agreeing that you  */
/* have read, and that you agree to comply with and are bound by, such        */
/* license terms. If you do not agree to be bound by the applicable license   */
/* terms, then you may not retain, install, activate or otherwise use the     */
/* software.                                                                  */
/*----------------------------------------------------------------------------*/

/** \file
* Example Source for NfcrdlibEx7_MIFAREPlus that demonstrates communication with MIFARE Plus EV1 contactless IC/MIFARE Plus EV2 contactless IC card.
* This example will detecting and activate Type A cards.
* Checks if the selected tag is MIFARE Plus EV1 and MIFARE Plus EV2 tag to proceed further with demo. If card is in SLO
* it demonstrates performing of Pre-personalization of the card then switches to SL1, performs SL1-SL3 mix mode
* operation and switches to SL3.
* For SL3 it does Read-Write EV0 and EV1 Secure Messaging Mode.
*
* Please refer Readme.txt file for Hardware Pin Configuration, Software Configuration and steps to build and
* execute the project which is present in the same project directory.
*
* $Author$
* $Revision$ (v06.10.00)
* $Date$
*
*/

/**
* Reader Library Headers
*/
#include <phApp_Init.h>
#include <NfcrdlibEx7_MIFAREPlus.h>


/* *****************************************************************************************************************
* MACROS/Defines
* ***************************************************************************************************************** */
#define KEYCOUNT                                0x05
/* Macros for various EEProm memory size for MFPEVx. ---------------------------------------------------------- */
#define EEPROM_2KB                              0x00U   /**< 2kB EEProm memory size */
#define EEPROM_4KB                              0x01U   /**< 4kB EEProm memory size */

/* Macros for switching between ISO14443 layer 3 or layer 4. ---------------------------------------------------*/
#define ISO14443_L3                             0x00    /* Option for ISO14443 Layer 3 protocol. */
#define ISO14443_L4                             0x01    /* Option for ISO14443 Layer 4 protocol. */

#define MIFARE_KEYTYPE_A                        0x0A    /**< MIFARE KeyType for key A. */
#define MIFARE_KEYTYPE_B                        0x0B    /**< MIFARE KeyType for key B. */

#define ENCRYPTED_OFF                           0x00    /**< Plain */
#define ENCRYPTED_ON                            0x01    /**< Encrypted */
#define MAC_RESPONSE_OFF                        0x00    /**< No MAC on response */
#define MAC_RESPONSE_ON                         0x01    /**< MAC on response */
#define MAC_COMMAND_OFF                         0x00    /**< No MAC on command*/
#define MAC_COMMAND_ON                          0x01    /**< MAC on command ON*/

#define FIRST_AUTH                              0x01    /**< First Authentication */

#define SECURITY_LEVEL_0                        0x01    /* Option for Security level 0. */
#define SECURITY_LEVEL_1                        0x02    /* Option for Security level 1. */
#define SECURITY_LEVEL_3                        0x04    /* Option for Security level 3. */

#define SECURITY_LEVEL_1_MIFARE                 0x00    /* Option for Mifare mode in Security level 1. */
#define SECURITY_LEVEL_1_MFP                    0x01    /* Option for MFP mode in Security level 1. */
#define SECURITY_LEVEL_3_MFP                    0x02    /* Option for MFP mode in Security level 1. */

#define EV0_SECURE_MESSAGING                    0x00
#define EV1_SECURE_MESSAGING                    0x01

/* PICC key address. -------------------------------------------------------------------------------------------*/
#define CARD_MASTER_KEY                         0x9000  /* Card Master key address */
#define CARD_CONFIGURATION_KEY                  0x9001  /* Card Configuration key address */
#define LEVEL_3_SWITCH_KEY                      0x9003  /* Level 3 Switch key address */
#define SL_1_CARD_AUTHENTICATION_KEY            0x9004  /* SL1 Card Authentication key address */
#define LEVEL_3_SECTOR_SWITCH_KEY               0x9006  /* Level 3 Sector Switch key address */
#define L1_L3_MIX_SECTOR_SWITCH_KEY             0x9007  /* Level 1, Level 3 Mix Sector Switch key address */

#define VC_PROXIMITY_KEY                        0xA001  /* Proximity key adress.*/
#define VC_SELECT_ENC_KEY                       0xA080  /* VC Select key address for encryption. */
#define VC_SELECT_MAC_KEY                       0xA081  /* VC Select key address for macking. */

#define MFP_CONFIGURATION_BLOCK                 0xB000  /* MFP Configuration Block address */
#define VCIID                                   0xB001  /* Virtual Card InstallationIdentifier. */
#define ATSBLOCK                                0xB002  /* ATS Block address */
#define FIELD_CONFIGURATION_BLOCK               0xB003  /* Field Configuration Block address */
#define ANTI_TEARING_CONFIGURATION_BLOCK        0xB004  /* Anti-Tearing Configuration Block address */

#define TMAC_KEY_1                              0xC000  /* TMAC key 1.*/
#define TMAC_CONFIG_KEY_1                       0xC001  /* TMAC configuration key 1.*/
#define TMAC_BLOCK_1                            0xC002  /* TMAC block 1. */
#define TMAC_COMMIT_READER_ID_1                 0xC003  /* TMAC commit reader ID block 1.*/
#define TMAC_CONFIG_BLOCK_1_0                   0xC004  /* TMAC config block 1.0 */
#define TMAC_CONFIG_BLOCK_1_1                   0xC005  /* TMAC config block 1.1 */

#define PHYSICAL_AES_START_ADDRESS              0x4000  /* Physical start address of AES key location */

/* Software keystore key address loation. ----------------------------------------------------------------------*/
#define CARD_MASTER_KEY_ADDRESS                 0x00    /* Card Master key address in keystore */
#define CARD_MASTER_KEY_VERSION                 0x00    /* Card Master key version in keystore */

#define LEVEL_3_SWITCH_KEY_ADDRESS              0x01    /* Level 3 Switch key address in keystore */
#define LEVEL_3_SWITCH_KEY_VERSION              0x00    /* Level 3 Switch key version in keystore */

#define L1_L3_MIX_SECTOR_SWITCH_KEY_ADDRESS     0x02    /* Level 1, Level 3 mix sector switch key address in keystore */
#define L1_L3_MIX_SECTOR_SWITCH_KEY_VERSION     0x00    /* Level 1, Level 3 mix sector switch key version in keystore */

#define MIFARE_KEY_ADDRESS_FF                   0x03    /* MIFARE key address in keystore */
#define MIFARE_KEY_VERSION_FF                   0x00    /* MIFARE key version in keystore */

#define AES_KEY_ADDRESS_FF                      0x04    /* AES key address in keystore */
#define AES_KEY_VERSION_FF                      0x00    /* AES key version in keystore */

/*******************************************************************************
**   Global Variable Declaration
*******************************************************************************/

phKeyStore_Sw_KeyEntry_t       aKeyEntry[KEYCOUNT];
phKeyStore_Sw_KeyVersionPair_t aKeyVersion[KEYCOUNT];
phKeyStore_Sw_KUCEntry_t       aKeyUsageCounter[KEYCOUNT];

phpalI14443p3a_Sw_DataParams_t     * ppalI14443p3a; /* Pointer to PAL 14443-3A data-params */
phpalI14443p4_Sw_DataParams_t      * ppalI14443p4;  /* Pointer to PAL 14443-4 data-params */
phpalI14443p4a_Sw_DataParams_t     * ppalI14443p4a; /* Pointer to PAL 14443-4 data-params */
phalMfpEVx_Sw_DataParams_t         * palMfpEVx;     /* Pointer to AL MIFAREPlusEv1 data-params */
phKeyStore_Sw_DataParams_t         * pKeyStore;     /* Pointer to Keystore data-params */


/* PICC and software keystore key values. ----------------------------------------------------------------------*/
uint8_t gaCardMasterKey[]               = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaCardConfigKey[]               = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaLevel3SwitchKey[]             = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaSL1CardAuthKey[]              = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaLevel3SectorSwitchKey[]       = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaLevel13SMixSectorSwitchKey[]  = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

uint8_t gaVcProximityKey[]              = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaVcSelectEncKey[]              = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaVcSelectMacKey[]              = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

uint8_t gaMfpCongifBlock[]              = { 0xFF, 0x0F, 0x00, 0xAA, 0xAA, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
//uint8_t gaVCIID[]                       = { 0xA0, 0x00, 0x00, 0x03, 0x96, 0x56, 0x43, 0x41, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
//uint8_t gaFieldConfigBlock[]            = { 0x00, 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaAntiTearingConfigBlock[]      = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };

uint8_t gaTMAC_Key_1[]                  = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaTMAC_ConfigKey_1[]            = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaTMAC_Block_1[]                = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t gaTMAC_Commit_Reader_ID_1[]     = { 0x01, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x01 };
uint8_t gaTMAC_Config_Block_1_0[]       = { 0x55, 0x55, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

uint8_t gaMifare_SectorKey_FF[]         = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x07, 0x80, 0x69, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
uint8_t gaAES_SectorKey_FF[]            = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

uint8_t gaValueBlock[]                  = { 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0xFF };

uint8_t bCurrentSecurityLevel = SECURITY_LEVEL_0;   /* Current security level of the card. */
uint8_t bEEPROMSize = EEPROM_4KB;                   /* Eeprom size. */

uint8_t aUid[10];                                   /* Complete Uid, 4, 7, 10 byte . */
uint8_t bUid_Len     = 0;                           /* Length of the UID received. */
uint8_t bSak;                                       /* Select Acknowledge data */
uint8_t aPcdCap2In[6];                              /* PCDCapsIn[0], 0x00: EV0 Secure messaging, 0x01: EV1 secure messaging. */
uint8_t aPcdCap2Out[6];                             /* Pointer to PCDCaps sent from the card (6 bytes). */
uint8_t aPdCap2[6];                                 /* Pointer to PDCaps. (6 bytes) */

uint16_t aSectorNo[]    = { 0x4001, 0x4003 };
uint16_t aKeyB_Key[]    = { AES_KEY_ADDRESS_FF, AES_KEY_ADDRESS_FF };
uint16_t aKeyB_Ver[]    = { AES_KEY_VERSION_FF, AES_KEY_VERSION_FF };
uint8_t  bProduct;

#ifdef PHOSAL_FREERTOS_STATIC_MEM_ALLOCATION
uint32_t MfPlusTaskBuffer[MIFAREPLUSEV1_TASK_STACK];
#else /* PHOSAL_FREERTOS_STATIC_MEM_ALLOCATION */
#define aMfPlusTaskBuffer    NULL
#endif /* PHOSAL_FREERTOS_STATIC_MEM_ALLOCATION */


/* *****************************************************************************************************************
* Types/Structure Declarations
* ***************************************************************************************************************** */

/* *****************************************************************************************************************
* Function Prototypes
* ***************************************************************************************************************** */
phStatus_t Test_MIFAREPlusEV1_Features();
phStatus_t Demo_SecurityLayer ( uint8_t bSecurityLevel, uint8_t bISOLayer, uint8_t bSecureMessaging);

/*******************************************************************************
**   Function Definitions
*******************************************************************************/
void PrintData ( uint8_t* pBuffer, uint32_t dwLength, char* pFormat, char* pSpecialChar )
{
    uint32_t dwIndex;

    for ( dwIndex = 0; dwIndex < dwLength; ++dwIndex )
    {
        DEBUG_PRINTF ( pFormat, pBuffer[dwIndex] );
    }
    DEBUG_PRINTF ( pSpecialChar );
}

phStatus_t ActivateCard ( uint8_t bIsoLayer )
{
    phStatus_t wStatus;
    uint8_t bCardPlaced = 0;
    uint8_t bMoreCardsAvailable;    /* more card available */
    uint8_t bCidEnabled;            /* Unequal '0' if Card Identifier is enabled. */
    uint8_t bCid = 1;               /* Card Identifier. */
    uint8_t bNadSupported;          /* Node Address Support; Unequal '0' if supported. */
    uint8_t bFwi;                   /* Frame Waiting Integer. */
    uint8_t bFsdi = 8;              /* PCD Frame Size Integer; 0-8 */
    uint8_t bFsci;                  /* PICC Frame Size Integer; 0-8. */

    uint8_t aAts[64];               /* Answer to Select; uint8_t[1-64] */

    /* Turn On the RF Field */
    PH_CHECK_SUCCESS_FCT ( wStatus, phhalHw_FieldOff ( pHal ) );

    /* Wait for RF ON Guard Time */
    PH_CHECK_SUCCESS_FCT ( wStatus, phhalHw_Wait ( pHal, PHHAL_HW_TIME_MILLISECONDS, 5U ) );

    /* Apply Type A Tech Protocol. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phhalHw_ApplyProtocolSettings(pHal, PHHAL_HW_CARDTYPE_ISO14443A));

    /* Turn On the RF Field */
    PH_CHECK_SUCCESS_FCT ( wStatus, phhalHw_FieldOn ( pHal ) );

    /* Wait for RF ON Guard Time */
    PH_CHECK_SUCCESS_FCT ( wStatus, phhalHw_Wait ( pHal, PHHAL_HW_TIME_MILLISECONDS, 5U ) );

    /* Activate Layer 3 card. In loop till a card is detected. */
    do
    {
        wStatus = phpalI14443p3a_ActivateCard (ppalI14443p3a, NULL, 0x00, aUid, &bUid_Len, &bSak, &bMoreCardsAvailable );

        if ( ( wStatus != PH_ERR_SUCCESS ) && ( !bCardPlaced ) )
        {
            DEBUG_PRINTF ( "Place Mifare Plus Ev1 card into field.\n" );
            bCardPlaced = 1;
        }
    } while ( wStatus != PH_ERR_SUCCESS );
    CHECK_SUCCESS ( wStatus );

    /* Set Auth mode as Iso14443 Level 3. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( palMfpEVx, PHAL_MFPEVX_AUTH_MODE, PHAL_MFPEVX_NOT_AUTHENTICATED_L3 ) );

    /* Activate L4 */
    if ( bIsoLayer )
    {
        PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p4a_ActivateCard ( ppalI14443p4a, bFsdi, bCid, PHPAL_I14443P4A_DATARATE_106, PHPAL_I14443P4A_DATARATE_106, aAts ) );

        /* Retrieve the ISO14443-4A protocol parameters. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p4a_GetProtocolParams( ppalI14443p4a, &bCidEnabled, &bCid, &bNadSupported, &bFwi,
            &bFsdi, &bFsci ) );

        /* Set the ISO14443-4 protocol parameters. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phpalI14443p4_SetProtocol( ppalI14443p4, bCidEnabled, bCid, bNadSupported, 0, bFwi, bFsdi, bFsci ) );

        /* Set Auth mode as Iso14443 Level 3. */
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_SetConfig ( palMfpEVx, PHAL_MFPEVX_AUTH_MODE, PHAL_MFPEVX_NOT_AUTHENTICATED_L4 ) );
    }

    return PH_ERR_SUCCESS;
}

phStatus_t SetSecurityLevel ()
{
    phStatus_t wStatus = 0;

    /* Activate the card in ISO14443 L3 to get the SAK value. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L3 ) );

    switch ( bSak )
    {
        case 0x18:
        case 0x08:
            bCurrentSecurityLevel = SECURITY_LEVEL_1;
            break;

        case 0x20:
            /* Activate the card in ISO14443 L4. */
            wStatus = ActivateCard ( ISO14443_L4 );
            if ( wStatus != PH_ERR_SUCCESS )
            {
                DEBUG_PRINTF ( " Not a valid MFP EV1/EV2 card\n" );
                return wStatus;
            }

            /* Perform Write perso to identify for Security Level 0. */
            wStatus = phalMfpEVx_WritePerso ( palMfpEVx, ISO14443_L4, CARD_MASTER_KEY, 1, gaCardMasterKey );
            if ( wStatus == PH_ERR_SUCCESS )
                bCurrentSecurityLevel = SECURITY_LEVEL_0;
            else
            {
                wStatus = phalMfpEVx_AuthenticateSL3 ( palMfpEVx, FIRST_AUTH, CARD_MASTER_KEY, CARD_MASTER_KEY_ADDRESS, CARD_MASTER_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 );

                if ( ( (wStatus & PH_ERR_MASK) == PHAL_MFPEVX_ERR_AUTH ) ||  ( wStatus == PH_ERR_SUCCESS ) )
                    bCurrentSecurityLevel = SECURITY_LEVEL_3;
                else
                {
                    DEBUG_PRINTF ( " Not a valid MFP EV1/EV2 card to be used with this Example Demo. \n" );
                    return wStatus;
                }
            }
            break;

        default:
            DEBUG_PRINTF ( " Not a valid MFP EV1/EV2 card\n" );
            return PH_ERR_UNSUPPORTED_PARAMETER;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t IncludeDemo ( uint8_t bSecurityLevel )
{
    if ( ( ( bCurrentSecurityLevel & bSecurityLevel ) != SECURITY_LEVEL_0 ) &&
        ( ( bCurrentSecurityLevel & bSecurityLevel ) != SECURITY_LEVEL_1 ) &&
        ( ( bCurrentSecurityLevel & bSecurityLevel ) != SECURITY_LEVEL_3 ) )
    {
        DEBUG_PRINTF ( "\n\n" );
        DEBUG_PRINTF ( "This demo feature is not supported in Security Level %d\n", ( ( bCurrentSecurityLevel == SECURITY_LEVEL_0 ) ? 0x00 : ( bCurrentSecurityLevel == SECURITY_LEVEL_1 ) ? 0x01 : 0x03) );
        DEBUG_PRINTF ( "\n\n" );

        return PH_ERR_UNSUPPORTED_COMMAND;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t AddKeyToKeyStore ( uint8_t bKeyNo, uint8_t bKeyVersion, uint8_t* pKey, uint16_t wKeyType)
{
    phStatus_t wStatus = 0;

    PH_CHECK_SUCCESS_FCT ( wStatus, phKeyStore_FormatKeyEntry ( pKeyStore, bKeyNo, wKeyType ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, phKeyStore_SetKeyAtPos ( pKeyStore, bKeyNo, 0x00, wKeyType, pKey, bKeyVersion ) );

    return PH_ERR_SUCCESS;
}

phStatus_t DetectProduct ()
{
    phStatus_t wStatus = 0;
    uint8_t aVerInfo[33];

    /* Activate Layer 4 card. */
    wStatus = ActivateCard ( ISO14443_L4 );
    CHECK_SUCCESS ( wStatus );

    /* Send Get Version command to MIFARE Plus EV1 and MIFARE Plus EV2 Card to get successful response. */
    wStatus = phalMfpEVx_GetVersion ( palMfpEVx, aVerInfo );
    if ((wStatus & PH_ERR_MASK) == PHAL_MFPEVX_ERR_CMD_INVALID)
    {
        DEBUG_PRINTF ( "Used card is MIFARE Plus Card. \n" );
        return wStatus;
    }
    else
    {
        CHECK_SUCCESS ( wStatus );
    }

    bEEPROMSize = (uint8_t) ( ( aVerInfo[5] == 0x16 ) ? EEPROM_2KB : EEPROM_4KB );
    bProduct = aVerInfo[10];

    return wStatus;
}

phStatus_t UpdateKeystore ()
{
    phStatus_t wStatus = 0;
    uint8_t aMifareKey[12];

    /* Add the required keys to keystore. */
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( CARD_MASTER_KEY_ADDRESS, CARD_MASTER_KEY_VERSION, gaCardMasterKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( LEVEL_3_SWITCH_KEY_ADDRESS, LEVEL_3_SWITCH_KEY_VERSION, gaLevel3SwitchKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( L1_L3_MIX_SECTOR_SWITCH_KEY_ADDRESS, L1_L3_MIX_SECTOR_SWITCH_KEY_VERSION, gaLevel13SMixSectorSwitchKey, PH_KEYSTORE_KEY_TYPE_AES128 ) );

    /* Update MIFARE keys to software keystore. */
    memcpy ( aMifareKey, gaMifare_SectorKey_FF, 6);
    memcpy ( &aMifareKey[6], &gaMifare_SectorKey_FF[10], 6);
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aMifareKey, PH_KEYSTORE_KEY_TYPE_MIFARE ) );

    /* Update AES keys to software keystore. */
    PH_CHECK_SUCCESS_FCT ( wStatus, AddKeyToKeyStore ( AES_KEY_ADDRESS_FF, AES_KEY_VERSION_FF, gaAES_SectorKey_FF, PH_KEYSTORE_KEY_TYPE_AES128 ) );

    return PH_ERR_SUCCESS;
}

uint8_t GetNumberofSectors ()
{

    switch ( bEEPROMSize )
    {
        case EEPROM_2KB:
            return ( 32 );

        case EEPROM_4KB:
            return ( 40 );

        default:
            return 0;
    }
}

phStatus_t GetAESSectorBlock_A ( uint16_t wSectorNr )
{
    return ( uint16_t ) ( ( wSectorNr * 2 ) + PHYSICAL_AES_START_ADDRESS );
}

phStatus_t GetAESSectorBlock_B ( uint16_t wSectorNr )
{
    return ( uint16_t ) ( ( ( wSectorNr * 2 ) + PHYSICAL_AES_START_ADDRESS ) + 1 );
}

phStatus_t ConfigureValueBlock ( uint8_t bBlockNo, uint8_t bSecurityLevel, uint8_t bISOLayer )
{
    phStatus_t wStatus      = 0;
    uint8_t aTMC[]          = { 0x00, 0x00, 0x00, 0x00 };
    uint8_t aTMV[]          = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( bISOLayer ) );

    /* Update the address field. */
    gaValueBlock[12] = bBlockNo;
    gaValueBlock[13] = ( uint8_t ) ( ~bBlockNo );
    gaValueBlock[14] = bBlockNo;
    gaValueBlock[15] = ( uint8_t ) ( ~bBlockNo );

    /* Authenticate according to the layer option. */
    switch ( bSecurityLevel )
    {
        case SECURITY_LEVEL_1_MIFARE:
            /* Authenticate the block using Mifare key A. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateMfc ( palMfpEVx, bBlockNo, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len ) );
            break;

        case SECURITY_LEVEL_1_MFP:
            /* Switch sector 0 and 1 to SL1 ans SL3 mixed mode. */
            wStatus = phalMfpEVx_SSAuthenticate ( palMfpEVx, 0x00, L1_L3_MIX_SECTOR_SWITCH_KEY, L1_L3_MIX_SECTOR_SWITCH_KEY_ADDRESS, L1_L3_MIX_SECTOR_SWITCH_KEY_VERSION,
                0, NULL, ( sizeof ( aSectorNo ) / 2 ), aSectorNo, aKeyB_Key, aKeyB_Ver, 0, NULL );

            /* Activate the card. */
            PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( ISO14443_L4 ) );

            /* Perform AES Sector authentication key. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( palMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A ( bBlockNo / 4 ), AES_KEY_ADDRESS_FF, AES_KEY_VERSION_FF,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        case SECURITY_LEVEL_3_MFP:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( palMfpEVx, FIRST_AUTH, GetAESSectorBlock_A ( bBlockNo / 4 ), AES_KEY_ADDRESS_FF, AES_KEY_VERSION_FF,
                0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;
    }

    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Write ( palMfpEVx, ENCRYPTED_OFF, MAC_COMMAND_OFF, bBlockNo, 1, gaValueBlock, aTMC, aTMV ) );

    return PH_ERR_SUCCESS;
}


phStatus_t Demo_PrePersonalization ( uint8_t bISOLayer )
{
    phStatus_t wStatus = 0;
    uint16_t wSector = 0;

    /* Activate to the respective ISO14443 layer. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( bISOLayer ) );

    /* Write the Card Master key. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, CARD_MASTER_KEY, 1, gaCardMasterKey ) );

    /* Write the Card configuration key. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, CARD_CONFIGURATION_KEY, 1, gaCardConfigKey ) );

        /* Write the Level 3 Switch key. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, LEVEL_3_SWITCH_KEY, 1, gaLevel3SwitchKey ) );

    /* Write the SL1 Card Auth key. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, SL_1_CARD_AUTHENTICATION_KEY, 1, gaSL1CardAuthKey ) );

    /* Write the Level 3 Sector Switch key. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, LEVEL_3_SECTOR_SWITCH_KEY, 1, gaLevel3SectorSwitchKey ) );

    /* Write the Level 1, Level 3 Mix Sector Switch key. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, L1_L3_MIX_SECTOR_SWITCH_KEY, 1, gaLevel13SMixSectorSwitchKey ) );

    /* Write the VC keys. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, VC_PROXIMITY_KEY, 1, gaVcProximityKey ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, VC_SELECT_ENC_KEY, 1, gaVcSelectEncKey ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, VC_SELECT_MAC_KEY, 1, gaVcSelectMacKey ) );

    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, MFP_CONFIGURATION_BLOCK, 1, gaMfpCongifBlock ) );
    /*PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso (palMfpEVx, bISOLayer, FIELD_CONFIGURATION_BLOCK, 1, gaFieldConfigBlock ) );*/
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, ANTI_TEARING_CONFIGURATION_BLOCK, 1, gaAntiTearingConfigBlock ) );

    /* Write the transaction MAC details. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, TMAC_KEY_1, 1, gaTMAC_Key_1 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, TMAC_CONFIG_KEY_1, 1, gaTMAC_ConfigKey_1 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, TMAC_BLOCK_1, 1, gaTMAC_Block_1 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, TMAC_COMMIT_READER_ID_1, 1, gaTMAC_Commit_Reader_ID_1 ) );
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, TMAC_CONFIG_BLOCK_1_0, 1, gaTMAC_Config_Block_1_0 ) );

    /* Write the AES sector keys. */
    for ( wSector = 0; wSector < GetNumberofSectors () * 2; wSector++ )
    {
        PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_WritePerso ( palMfpEVx, bISOLayer, (uint16_t) ( 0x4000 + wSector ), 1, gaAES_SectorKey_FF ) );
    }

    /* Set the hardware timeout to 30ms. */
    wStatus = phhalHw_SetConfig(pHal, PHHAL_HW_CONFIG_TIMEOUT_VALUE_US, 30000);
    CHECK_SUCCESS ( wStatus );

    /* Commit Perso to switch to security Level 1. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_CommitPerso ( palMfpEVx, 0x01, bISOLayer ) );

    /* Activate to the respective ISO14443 layer. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( bISOLayer ) );

    return PH_ERR_SUCCESS;
}


phStatus_t Demo_SecurityLayer ( uint8_t bSecurityLevel, uint8_t bISOLayer, uint8_t bSecureMessaging )
{
    phStatus_t wStatus          = 0;
    uint8_t bData_BlockNo       = 0x01;
    uint8_t bValue_BlockNo      = 0x02;
    uint8_t bAddress            = 0;

    uint8_t aTMC[]              = { 0x00, 0x00, 0x00, 0x00 };
    uint8_t aTMV[]              = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

    uint8_t aRx_Data[]          = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    uint8_t aTx_Data[]          = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10 };

    uint8_t aRx_Value[]         = { 0x00, 0x00, 0x00, 0x00 };
    uint8_t aTx_Value[]         = { 0x00, 0x00, 0x00, 0x00 };

    /* Activate the card. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ActivateCard ( bISOLayer ) );

    /* Configure a block as value block. */
    PH_CHECK_SUCCESS_FCT ( wStatus, ConfigureValueBlock ( bValue_BlockNo, bSecurityLevel, bISOLayer ) );

    DEBUG_PRINTF ( "Non TMAC feature on Data block %04X ------------------------------------------\n", bData_BlockNo );

    /* Authenticate according to the layer option. */
    switch ( bSecurityLevel )
    {
        case SECURITY_LEVEL_1_MIFARE:
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateMfc ( palMfpEVx, bData_BlockNo, MIFARE_KEYTYPE_A, MIFARE_KEY_ADDRESS_FF, MIFARE_KEY_VERSION_FF, aUid, bUid_Len ) );
            break;

        case SECURITY_LEVEL_1_MFP:
            /* Perform authentication using AES Setor. */
            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL1 ( palMfpEVx, ISO14443_L4, FIRST_AUTH, GetAESSectorBlock_A ( ( uint16_t) ( bData_BlockNo / 4 ) ),
                AES_KEY_ADDRESS_FF, AES_KEY_VERSION_FF, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;

        case SECURITY_LEVEL_3_MFP:
            /* Update the secure messaging. */
            aPcdCap2In[0] = bSecureMessaging;

            PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_AuthenticateSL3 ( palMfpEVx, FIRST_AUTH, GetAESSectorBlock_A ( ( uint16_t) ( bData_BlockNo / 4 ) ),
                AES_KEY_ADDRESS_FF, AES_KEY_VERSION_FF, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 ) );
            break;
    }

    /* Write a random data to the block. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Write ( palMfpEVx, ENCRYPTED_OFF, MAC_COMMAND_OFF, bData_BlockNo, 1, aTx_Data, aTMC, aTMV ) );
    DEBUG_PRINTF ( "(Write Data) Tx Value             :" );
    PrintData ( aTx_Data, sizeof ( aTx_Data ), "%02X ", "\n" );

    /* Read the data written from the block. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_Read ( palMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bData_BlockNo, 1, aRx_Data ) );
    DEBUG_PRINTF ( "(Read Data) Rx Value              :" );
    PrintData ( aRx_Data, sizeof ( aRx_Data ), "%02X ", "\n" );
    DEBUG_PRINTF ( "\n" );

    DEBUG_PRINTF ( "Non TMAC feature on Value block %04X -----------------------------------------\n", bValue_BlockNo );

    /* Increment transfer the data to the block. */
    aTx_Value[0] = 0x01;
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_IncrementTransfer ( palMfpEVx, MAC_COMMAND_OFF, bValue_BlockNo, bValue_BlockNo, aTx_Value, aTMC, aTMV ) );
    DEBUG_PRINTF ( "(IncrementTransfer Value) Tx Value:" );
    PrintData ( aTx_Value, sizeof ( aTx_Value ), "%02X ", "\n" );

    /* Read the value written from the block. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_ReadValue ( palMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bValue_BlockNo, aRx_Value, &bAddress ) );
    DEBUG_PRINTF ( "(Read Value) Rx Value             :" );
    PrintData ( aRx_Value, sizeof ( aRx_Value ), "%02X ", "\n" );

    /* Decrement transfer the data to the block. */
    aTx_Value[0] = 0x01;
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_DecrementTransfer ( palMfpEVx, MAC_COMMAND_OFF, bValue_BlockNo, bValue_BlockNo, aTx_Value, aTMC, aTMV ) );
    DEBUG_PRINTF ( "(DecrementTransfer Value) Tx Value:" );
    PrintData ( aTx_Value, sizeof ( aTx_Value ), "%02X ", "\n" );

    /* Read the value written from the block. */
    PH_CHECK_SUCCESS_FCT ( wStatus, phalMfpEVx_ReadValue ( palMfpEVx, ENCRYPTED_ON, MAC_RESPONSE_ON, MAC_COMMAND_ON, bValue_BlockNo, aRx_Value, &bAddress ) );
    DEBUG_PRINTF ( "(Read Value) Rx Value             :" );
    PrintData ( aRx_Value, sizeof ( aRx_Value ), "%02X ", "\n" );

    return PH_ERR_SUCCESS;
}


/*******************************************************************************
**   Main Function
*******************************************************************************/
int main (void)
{
    do
    {
        phStatus_t status = PH_ERR_INTERNAL_ERROR;
        phNfcLib_Status_t     dwStatus;

#ifdef PH_PLATFORM_HAS_ICFRONTEND
        phNfcLib_AppContext_t AppContext = {0};
#endif /* PH_PLATFORM_HAS_ICFRONTEND */

#ifndef PH_OSAL_NULLOS
        phOsal_ThreadObj_t MfPlus;
#endif /* PH_OSAL_NULLOS */

        /* Perform Controller specific initialization. */
        phApp_CPU_Init();

        /* Perform OSAL Initialization. */
        (void)phOsal_Init();

        /* Print Example application name */
        DEBUG_PRINTF("\n *** MIFARE Plus EV1 and MIFARE Plus EV2 Example *** \n");

#ifdef PH_PLATFORM_HAS_ICFRONTEND
        status = phbalReg_Init(&sBalParams, sizeof(phbalReg_Type_t));
        CHECK_STATUS(status);

        AppContext.pBalDataparams = &sBalParams;
        dwStatus = phNfcLib_SetContext(&AppContext);
        CHECK_NFCLIB_STATUS(dwStatus);
#endif /* PH_PLATFORM_HAS_ICFRONTEND */

        /* Initialize library */
        dwStatus = phNfcLib_Init();
        CHECK_NFCLIB_STATUS(dwStatus);
        if(dwStatus != PH_NFCLIB_STATUS_SUCCESS)
            break;

        /* Set the generic pointer */
        pHal = phNfcLib_GetDataParams(PH_COMP_HAL);

        ppalI14443p3a = (phpalI14443p3a_Sw_DataParams_t *) phNfcLib_GetDataParams(PH_COMP_PAL_ISO14443P3A);
        ppalI14443p4 = (phpalI14443p4_Sw_DataParams_t *) phNfcLib_GetDataParams(PH_COMP_PAL_ISO14443P4);
        ppalI14443p4a = (phpalI14443p4a_Sw_DataParams_t *) phNfcLib_GetDataParams(PH_COMP_PAL_ISO14443P4A);
        pKeyStore = (phKeyStore_Sw_DataParams_t *)  phNfcLib_GetDataParams(PH_COMP_KEYSTORE);
        palMfpEVx = (phalMfpEVx_Sw_DataParams_t *)  phNfcLib_GetDataParams(PH_COMP_AL_MFPEVX);

        /* Configure the IRQ */
        status = phApp_Configure_IRQ();
        CHECK_STATUS(status);
        if(status != PH_ERR_SUCCESS) break;

#ifndef PH_OSAL_NULLOS

        MfPlus.pTaskName = (uint8_t *) "MfPlusEV1";
        MfPlus.pStackBuffer = aMfPlusTaskBuffer;
        MfPlus.priority = MIFAREPLUSEV1_TASK_PRIO;
        MfPlus.stackSizeInNum = MIFAREPLUSEV1_TASK_STACK;
        phOsal_ThreadCreate(&MfPlus.ThreadHandle, &MfPlus, &MIFAREPlusEV1_EV2_Demo, NULL);

        /* Start Scheduler */
        phOsal_StartScheduler();

        /* Print Error info */
        DEBUG_PRINTF("RTOS Error : Scheduler exited. \n");

#else
        (void)MIFAREPlusEV1_EV2_Demo(NULL);
#endif
    }  while(0);

    while(1); //Comes here if initialization failure or scheduler exit due to error

    return 0;
}

/***********************************************************************************************
* \brief   This function demonstrates the MIFARE Plus EV1 application as mentioned below
*           1. Detects and Activates MIFAREPlus Ev1/Ev2 card
*           2. Check the security level of the cards and does the following depending upon the
*           card security level
*           - For card in Securtity Level 0, it does the pre-personalization of the card
*           and changes the security level to SL1-SL3 mix mode
*           - For Card in SL1-SL3 mix mode, it does the read-write operations in SL1 and SL3
*           and switches card security level to SL3
*           - For Card in SL3, it does read-write in EV0 and EV1 secure messaging mode.

* \param   *pParams
* \return  This function will never return
**********************************************************************************************/
void MIFAREPlusEV1_EV2_Demo(void *pParams)
{
    phStatus_t  status = 0;
    uint8_t aAtqa[2];

    /* init. keystore */
    status = phKeyStore_Sw_Init(pKeyStore,
        sizeof(phKeyStore_Sw_DataParams_t),
        aKeyEntry,
        KEYCOUNT,
        aKeyVersion,
        0x01,
        aKeyUsageCounter,
        KEYCOUNT);
    CHECK_STATUS(status);

    /* Add keys to the software keystore component. */
    status = UpdateKeystore ();
    CHECK_STATUS ( status );

    /* Continuous loop */
    while(1)
    {
        DEBUG_PRINTF("\nReady to detect\n");

        ActivateCard(ISO14443_L4);

        status = DetectProduct();
        DEBUG_ERROR_PRINT(status);

        if (((bProduct != 0x02) && (bProduct != 0x01)) && (status != PH_ERR_SUCCESS))
        {
            DEBUG_PRINTF("\n Not MIFAREPlus EV1 OR MIFAREPlus EV2 card \n");
            continue;
        }

        status = Test_MIFAREPlusEV1_Features();
        DEBUG_ERROR_PRINT(status);

        DEBUG_PRINTF("\nPlease Remove the Card\n\n");

        /* Field reset */
        status = phhalHw_FieldReset(pHal);
        DEBUG_ERROR_PRINT(status);

        /* Make sure that example application is not detecting the same card continuously. */
        do
        {
            /* Send WakeUpA */
            status = phpalI14443p3a_WakeUpA(ppalI14443p3a, aAtqa);
            /* Check for Status */
            if (status != PH_ERR_SUCCESS)
            {
                break; /* Card Removed, break from the loop */
            }

            /* Send HaltA */
            (void)phpalI14443p3a_HaltA(ppalI14443p3a);

            /* Delay of 5ms*/
            status = phhalHw_Wait(pHal, PHHAL_HW_TIME_MILLISECONDS, 5);
            DEBUG_ERROR_PRINT(status);
        }while(1);

        (void)phpalI14443p4_ResetProtocol(ppalI14443p4);
    }
}

phStatus_t Test_MIFAREPlusEV1_Features(void)
{
    phStatus_t wStatus;
    /* Set the current security level of the card. */
    wStatus = SetSecurityLevel ();
    CHECK_SUCCESS ( wStatus );

    /* Reset the authentication state. */
    wStatus = phalMfpEVx_ResetSecMsgState ( palMfpEVx );
    CHECK_SUCCESS ( wStatus );

    DEBUG_PRINTF ( "\n\n" );

    switch ( bCurrentSecurityLevel )
    {
        case SECURITY_LEVEL_0:
            DEBUG_PRINTF ( "Card is in Security Level 0\n" );
            DEBUG_PRINTF ( "Doing Pre-personalization of card. ---------------------------------------------\n" );

            /* Check if this feature is valid in the current security level of the card. */
            wStatus = IncludeDemo ( SECURITY_LEVEL_0 );

            if ( wStatus == PH_ERR_SUCCESS )
            {
                /* Prepersonalize the card. */
                wStatus = Demo_PrePersonalization ( ISO14443_L4 );
                CHECK_SUCCESS ( wStatus );

                DEBUG_PRINTF ( "Pre-personalization Completed.\n" );
            }
            break;

        case SECURITY_LEVEL_1:
            DEBUG_PRINTF ( "Card is in Security Level 1\n" );
            DEBUG_PRINTF ( "SL1 SL3 mixed mode operation. ---------------------------------------------- \n\n" );

            /* Check if this feature is valid in the current security level of the card. */
            wStatus = IncludeDemo ( SECURITY_LEVEL_1 );

            if ( wStatus == PH_ERR_SUCCESS )
            {
                /* Demo with SL1  . */
                DEBUG_PRINTF ( "Command sequence in Security level 1-------------------------------------- \n" );
                wStatus = Demo_SecurityLayer ( SECURITY_LEVEL_1_MFP, ISO14443_L4, EV0_SECURE_MESSAGING );
                CHECK_SUCCESS ( wStatus );
                DEBUG_PRINTF ( "\n" );

                DEBUG_PRINTF ( "Command sequence in Security level 3-------------------------------------- \n" );
                /* Demo with SL3  . */
                wStatus = Demo_SecurityLayer ( SECURITY_LEVEL_3_MFP, ISO14443_L4, EV0_SECURE_MESSAGING );
                CHECK_SUCCESS ( wStatus );

                DEBUG_PRINTF ( "\n\n" );
            }

            DEBUG_PRINTF ( "Switching to Security Level 3. --------------------------------------------- \n" );

            /* Check if this feature is valid in the current security level of the card. */
            wStatus = IncludeDemo ( SECURITY_LEVEL_1 );

            if ( wStatus == PH_ERR_SUCCESS )
            {
                /* Activate to ISO14443 L4. */
                wStatus = ActivateCard ( ISO14443_L4 );
                CHECK_SUCCESS ( wStatus );

                /* Switch to security level 3. */
                wStatus = phalMfpEVx_AuthenticateSL1 ( palMfpEVx, ISO14443_L4, FIRST_AUTH, LEVEL_3_SWITCH_KEY, LEVEL_3_SWITCH_KEY_ADDRESS,
                    LEVEL_3_SWITCH_KEY_VERSION, 0, NULL, 6, aPcdCap2In, aPcdCap2Out, aPdCap2 );
                CHECK_SUCCESS ( wStatus );

                DEBUG_PRINTF ( "Card has switched to Security Level 3\n" );
            }
            break;

        case SECURITY_LEVEL_3:
            DEBUG_PRINTF ( "Card is in Security Level 3\n\n" );
            DEBUG_PRINTF ( "Read and Write with EV0 Secure Messaging Mode. ----------------------------- \n" );

            /* Check if this feature is valid in the current security level of the card. */
            wStatus = IncludeDemo ( SECURITY_LEVEL_3 );

            if ( wStatus == PH_ERR_SUCCESS )
            {
                wStatus = Demo_SecurityLayer ( SECURITY_LEVEL_3_MFP, ISO14443_L4, EV0_SECURE_MESSAGING );
                CHECK_SUCCESS ( wStatus );

                DEBUG_PRINTF ( "\n\n" );
            }

            DEBUG_PRINTF ( "Read and Write with EV1 Secure Messaging Mode. ----------------------------- \n" );

            wStatus = Demo_SecurityLayer ( SECURITY_LEVEL_3_MFP, ISO14443_L4, EV1_SECURE_MESSAGING );
            CHECK_SUCCESS ( wStatus );

            DEBUG_PRINTF ( "\n\n" );
            break;

        default:
            break;
    }

    return PH_ERR_SUCCESS;
}

#ifdef NXPBUILD__PHHAL_HW_TARGET
/* Stubbed definitions in case TARGET is enabled */
uint8_t  sens_res[2]     = {0x04, 0x00};
uint8_t  nfc_id1[3]      = {0xA1, 0xA2, 0xA3};
uint8_t  sel_res         = 0x40;
uint8_t  nfc_id3         = 0xFA;
uint8_t  poll_res[18]    = {0x01, 0xFE, 0xB2, 0xB3, 0xB4, 0xB5,
                                   0xB6, 0xB7, 0xC0, 0xC1, 0xC2, 0xC3,
                                   0xC4, 0xC5, 0xC6, 0xC7, 0x23, 0x45 };
#endif /* NXPBUILD__PHHAL_HW_TARGET */

/***********************************************************************************************
*                            End Of File
**********************************************************************************************/
