/**
 * Copyright 2021 NXP
 *
 * This software is owned or controlled by NXP and may only be used
 * strictly in accordance with the applicable license terms.  By expressly
 * accepting such terms or by downloading, installing, activating and/or
 * otherwise using the software, you are agreeing that you have read, and
 * that you agree to comply with and are bound by, such license terms.  If
 * you do not agree to be bound by the applicable license terms, then you
 * may not retain, install, activate or otherwise use the software.
 *
 */

/** @file
 *
 * alpar.c:
 *
 *
 * Project:  PN7462AU
 *
 * $Date: 2021-09-23 15:00:00 +0200 (Thu, 23 Sep 2021) $
 * $Author: Petar Hlad (MobileKnowledge) $
 * $Revision:
 */

#include "alpar.h"
#include "ph_Log.h"
#include "phpalCt.h"
#include "phhalCt.h"
#include "phhalHif.h"
#include "phUser.h"
#include "phpalCt_T1.h"

uint8_t cmd_data_buffer[100];
uint8_t resp_data_buffer[100];
uint16_t cmd_buffer_len;
uint16_t resp_buffer_len;

static uint8_t bWI = 0x0A;             /**< Used for setting WI value for WWT calculation for T0 protocol.*/
static uint8_t bBWI = 0x04;            /**< Used for setting BWI value for BWT calculation for T1 protocol.*/
static uint8_t bCWI = 0x0D;            /**< Used for setting CWI value for CWT calculation for T1 protocol.*/
static uint8_t cardActivated = 0x00;
static uint8_t negotiableMode = 0x00;

extern void tx_response(uint8_t ack, uint8_t cmd, uint8_t *data, uint8_t data_length);
extern void phEx_HsuHifConfigBaudRate(phhalHif_Hsu_BaudRate_t baud_rate);
extern phStatus_t phEMVCo_LoopbackProcess(void);

extern uint8_t gphhalCt_BEmvEn;

/**
 * Structs
 */

typedef phStatus_t (*VOID_FUNCTION_PTR)(uint8_t*, uint16_t*, uint8_t*, uint16_t*);

typedef struct
{
   uint8_t              command;
   VOID_FUNCTION_PTR    function;
   uint8_t              CardNeeded;
   uint8_t              paramLen;
}  COMMAND_FUNCTION_STRUCT;

typedef struct
{
    uint16_t    phStatus_err;
    uint8_t     alpar_err;
} ALPAR_ERR_STRUCT;


static const COMMAND_FUNCTION_STRUCT OPCODE_FUNCTION_TABLE[] =
{
   {  ALPAR_CMD_CARD_COMMAND,           alpar_card_command,             1, 254},
   {  ALPAR_CMD_POWER_UP_5V,            alpar_power_up_5V,              1,   1},
   {  ALPAR_CMD_POWER_UP_3V,            alpar_power_up_3V,              1,   1},
   {  ALPAR_CMD_POWER_UP_1V8,           alpar_power_up_1V8,             1,   1},
   {  ALPAR_CMD_GET_FW_VERSION,         alpar_get_fw_version,           0,   0},
   {  ALPAR_CMD_POWER_OFF,              alpar_power_off,                0,   0},
   {  ALPAR_CMD_NEGOTIATE_PPS,          alpar_negotiate_pps,            1,   2},
   {  ALPAR_CMD_SET_CLOCK_CARD,         alpar_set_clock_card,           1,   1},
   {  ALPAR_CMD_SET_CARD_BAUD_RATE,     alpar_set_card_baud_rate,       1,   1},
   {  ALPAR_CMD_SET_NAD,                alpar_set_nad,                  0,   1},
   {  ALPAR_CMD_CHECK_PRES_CARD,        alpar_check_pres_card,          0,   0},
   {  ALPAR_CMD_SET_SERIAL_BAUD_RATE,   alpar_set_serial_baud_rate,     0,   1},
   {  ALPAR_CMD_GET_READER_STATUS,      alpar_get_reader_status,        0,   0},
   {  ALPAR_CMD_SHOW_FIDI,              alpar_show_fidi,                0,   0},
   {  ALPAR_CMD_START_EMV_LOOPBACK,     alpar_start_EMV_loopback,       0,   0},
   {  ALPAR_CMD_IDLE_MODE,              alpar_idle_mode,                1,   2},

   {  ALPAR_OUTGOING_CMD_CARD_PRES,     alpar_outgoing_card_presence,   0,   1},

   {  ALPAR_CMD_BAD_OPCODE,             0,                              0,   0},
};

phStatus_t alpar_handle_command(uint8_t cmd, uint8_t *data, uint16_t datalength)
{

    aStatus_t aStatus = ALPAR_ERR_SUCCESS;

    uint8_t idx = 0;

    resp_buffer_len = 0;

    while( (OPCODE_FUNCTION_TABLE[idx].command != cmd) &&
           (OPCODE_FUNCTION_TABLE[idx].command != ALPAR_CMD_BAD_OPCODE) )
    {
        idx++;
    }

    if (OPCODE_FUNCTION_TABLE[idx].command == ALPAR_CMD_BAD_OPCODE)
    {
        aStatus = ALPAR_ERR_UNKNOWN_COMMAND;
    }

    if (OPCODE_FUNCTION_TABLE[idx].CardNeeded)
    {
        if( (PH_ERR_CT_MAIN_CARD_PRESENT | PH_COMP_HAL_CT) != phhalCt_CheckCardPres() )
        {
            aStatus = ALPAR_ERR_CARD_ABSENT;
        }
    }

    if (cmd == ALPAR_CMD_CARD_COMMAND)
    {
        if (datalength > OPCODE_FUNCTION_TABLE[idx].paramLen)
        {
            aStatus = ALPAR_ERR_BAD_PARAMETER;
        }
    }
    else if (datalength != OPCODE_FUNCTION_TABLE[idx].paramLen)
    {
           aStatus = ALPAR_ERR_BAD_PARAMETER;
    }


    if(aStatus == ALPAR_ERR_SUCCESS)
    {

        cmd_buffer_len = datalength;
        resp_buffer_len = 0;

        memset(cmd_data_buffer, 0, 100);
        memset(resp_data_buffer, 0, 100);

        memcpy(cmd_data_buffer, data, datalength);

        aStatus = OPCODE_FUNCTION_TABLE[idx].function(cmd_data_buffer, &cmd_buffer_len, resp_data_buffer, &resp_buffer_len);

    }

    if(aStatus != ALPAR_ERR_SUCCESS)
    {
        if( resp_buffer_len == 0 )
        {

            resp_data_buffer[0] = aStatus;
            resp_buffer_len = 1;

            tx_response(0, cmd, resp_data_buffer, resp_buffer_len);

        }
        else
        {

            memcpy(&resp_data_buffer[1], &resp_data_buffer[0], resp_buffer_len);

            resp_data_buffer[0] = aStatus;
            resp_buffer_len++;

            tx_response(0, cmd, resp_data_buffer, resp_buffer_len);

        }

    }
    else
    {
        if( cmd != ALPAR_CMD_SET_SERIAL_BAUD_RATE)
        {
            tx_response(1, cmd, resp_data_buffer, resp_buffer_len);
        }
    }

    return PH_ERR_SUCCESS;
}

phStatus_t alpar_card_command(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Card Command\n");

    phStatus_t phStatus = PH_ERR_FAILED;

    if( *cmd_len < 2 )
    {
        return ALPAR_ERR_TOO_SHORT_APDU;
    }

    LOG_AU8("APDU Send:\n",cmd_buff, *cmd_len);

    if( cardActivated == 0x00 )
    {
        return ALPAR_ERR_CARD_DEACTIVATED;
    }

    /* Send the Apdu */
    phStatus = phpalCt_Transceive( cmd_buff, *cmd_len, resp_buff, resp_len);

    switch(phStatus)
    {

        case PH_ERR_SUCCESS:

            LOG_AU8("APDU Receive:\n",resp_buff, *resp_len);

            return ALPAR_ERR_SUCCESS;

        case PH_ERR_OPERATION_TIMEDOUT:
            //Api timed out.

            return ALPAR_ERR_TIMEOUT;

        case PH_COMP_HAL_CT | PH_ERR_INVALID_PARAMETER:
            //Parameters are invalid.

            return ALPAR_ERR_BAD_PARAMETER;

        case PH_COMP_HAL_CT | PH_ERR_CT_MAIN_CARD_ABSENT:
        case PH_COMP_HAL_CT | PH_ERR_CT_CARD_REMOVED:
            //Card is absent in the slot.

            return ALPAR_ERR_CARD_ABSENT;

        case PH_COMP_HAL_CT | PH_ERR_CT_MUTE_ERROR:
            //Card is muted.

            return ALPAR_ERR_CARD_MUTE_T1_EXG;

        case PH_COMP_HAL_CT | PH_ERR_CT_PARITY_ERROR:
            //Card has parity error while receiving the ATR.

            return ALPAR_ERR_PARITY_ERR_RX;

        case PH_COMP_HAL_CT | PH_ERR_CT_OVERUN_ERROR:
            //Fifo is over run while receiving the ATR.

            return ALPAR_ERR_CARD_OVERFLOW;

        case PH_COMP_HAL_CT | PH_ERR_CT_FRAMING_ERROR:
            //Framing error while receiving the ATR.

            return ALPAR_ERR_FRAMING_ERR;

        case PH_COMP_HAL_CT | PH_ERR_CT_TEMPERATURE_LATCHED:
            //This error bit is set when a temperature latch occurs.

            return ALPAR_ERR_TEMPERATURE_ALARM;

        case PH_COMP_HAL_CT | PH_ERR_CT_PROTECTION_LATCHED:
            //This error bit is set when a protection is latched.

            return ALPAR_ERR_SPLY_VOLTAGE_DROPOFF;

        case PH_COMP_PAL_CT | PH_ERR_CT_DADSAD_NOT_SUPPORTED:

            return ALPAR_ERR_BAD_NAD;

        case PH_COMP_PAL_CT | PH_ERR_CT_CHAINING_ABORT_REQUESTED:

            return ALPAR_ERR_CHAIN_ABORTED;

        case PH_COMP_PAL_CT | PH_ERR_CT_RESYNCH_SUCCESS:

            return ALPAR_ERR_RESYNC;

        case PH_COMP_PAL_CT | PH_ERR_CT_EXT_APDU_NOT_SUPPORTED:

            return ALPAR_ERR_WRONG_APDU;

        case PH_COMP_PAL_CT | PH_ERR_CT_RETRY_COUNT_FAILURE:
            //If Retry count has been exceeded than allowable value.

            return ALPAR_ERR_BAD_LRC;

        case PH_COMP_PAL_CT | PH_ERR_CT_INS_COMMAND_ERROR:
            //Improper INS is received from card.

            return ALPAR_ERR_PROCEDURE_BYTE_ERR;

        default:

            // TODO: return generic error code.
            ;

    }

    return ALPAR_ERR_SUCCESS;

}


phStatus_t alpar_check_pres_card(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Check Card presence\n");

    if ((PH_ERR_CT_MAIN_CARD_PRESENT | PH_COMP_HAL_CT) == phhalCt_CheckCardPres())
        resp_buff[0] = 0x01;
    else
        resp_buff[0] = 0x00;

    *resp_len = 1;

    return ALPAR_ERR_SUCCESS;

}


phStatus_t alpar_start_EMV_loopback(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("EMV Loopback\n");

    phEMVCo_LoopbackProcess();

    return ALPAR_ERR_SUCCESS;

}


phStatus_t alpar_power_off(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Power Off\n");

    phhalCt_CardDeactivate();

    cardActivated = 0x00;

    return ALPAR_ERR_SUCCESS;

}

phStatus_t alpar_get_fw_version(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    /* Identifies the SW version in ASCII characters */
    LOG_TXT("Get FW Version\n");

    char fw_version[] = "7462 120";
    strcpy((char*) resp_buff, fw_version);

    *resp_len = strlen(fw_version);

    return ALPAR_ERR_SUCCESS;

}


phStatus_t alpar_set_card_baud_rate(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Set Card Baud Rate\n");

    phStatus_t phStatus = PH_ERR_FAILED;

    phStatus = phhalCt_SetBaudRate( cmd_buff[0] , bWI, bBWI, bCWI );

    switch (phStatus)
    {

        case PH_ERR_SUCCESS:

            return ALPAR_ERR_SUCCESS;

        case PH_COMP_HAL_CT | PH_ERR_INVALID_PARAMETER:

            return ALPAR_ERR_BAD_FIDI;

        default:

            // TODO: return generic error code.
            ;

    }

    return ALPAR_ERR_SUCCESS;

}

phStatus_t alpar_set_serial_baud_rate(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    /* Change the baud rate onto the serial link between the host and the interface card */

    LOG_TXT("Set Card BaudRate\n");

    if ( cmd_buff[0] < 0 || 0x0D < cmd_buff[0] )
    {
        /* Wrong baud rate */
        return ALPAR_ERR_BAD_PARAMETER;
    }

    /* The command shall be acknowledged before changing the baud rate */
    tx_response(1, ALPAR_CMD_SET_SERIAL_BAUD_RATE, resp_buff, *resp_len);

    phUser_Wait(1000);

    phEx_HsuHifConfigBaudRate( (phhalHif_Hsu_BaudRate_t) cmd_buff[0]);

    return ALPAR_ERR_SUCCESS;

}


phStatus_t alpar_show_fidi(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT(" Show FiDi\n");

    resp_buff[0] = gphhalCt_BCurrentFiDi;

    *resp_len = 1;

    return ALPAR_ERR_SUCCESS;

}

phStatus_t alpar_negotiate_pps(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Negotiate PPS \n");

    if (negotiableMode == 0x00)
    {
        return ALPAR_ERR_NON_NEGOTIABLE_MODE;
    }

    phStatus_t phStatus = PH_ERR_FAILED;

    phhalCt_PPSParamType_t  sPPSParams;

    sPPSParams.bBWI = bBWI;
    sPPSParams.bCWI = bCWI;
    sPPSParams.bWI = bWI;

    sPPSParams.bNegotiableProtocol = 0x00;

    if(cmd_buff[0] == 0x00)
    {
        sPPSParams.bProtSelT0 = 0x01;
        sPPSParams.bProtSelT1 = 0x00;
    }
    else if(cmd_buff[0] == 0x01)
    {
        sPPSParams.bProtSelT0 = 0x00;
        sPPSParams.bProtSelT1 = 0x01;
    }
    else
    {
        return ALPAR_ERR_UNKNOWN_PROTOCOL;
    }

    sPPSParams.bFiDi = cmd_buff[1];

    phStatus = phhalCt_PPSRequestHandling(&sPPSParams);

    switch(phStatus)
    {

        case PH_ERR_SUCCESS:
        case PH_COMP_HAL_CT | PH_ERR_PPS_EXCHANGE_NOT_REQUIRED:

            return ALPAR_ERR_SUCCESS;

        case PH_COMP_HAL_CT | PH_ERR_INVALID_PARAMETER:

            return ALPAR_ERR_BAD_PARAMETER;

        case PH_COMP_HAL_CT | PH_ERR_CT_PPS_EXCHANGE_ERROR:

            return ALPAR_ERR_PPS_BAD_ANSWER;

        default:

            return ALPAR_ERR_PPS_NOT_ACCEPTED;

    }

}


phStatus_t alpar_set_clock_card(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Set Card Clock\n");

    const uint16_t clock_freqs[8] = { 27120, 13560, 9040, 6780, 5420, 4520, 3390, 1690 };
    const uint16_t max_clock_freqs[16] = {
            4000, 5000, 6000,  8000, 12000, 16000, 20000, 0,
               0, 5000, 7500, 10000, 15000, 20000,     0, 0
    };

    if( ( cmd_buff[0] >= 0 ) && ( cmd_buff[0] <= 7 ) )
    {

        uint8_t currentFi = gphhalCt_BCurrentFiDi >> 4;

        if( clock_freqs[ cmd_buff[0] ] <= max_clock_freqs[ currentFi ]  )
        {
            PH_REG_SET_FIELD(CT_CCRx_REG, ACC2_ACC0, cmd_buff[0]);
        }
        else
        {
            return ALPAR_ERR_CLK_FREQ_NOT_ACCEPTED;
        }

    }
    else
    {
        return ALPAR_ERR_BAD_PARAMETER;
    }

    return ALPAR_ERR_SUCCESS;
}

phStatus_t alpar_set_nad(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Set NAD\n");

    phStatus_t phStatus = PH_ERR_FAILED;

    phStatus = phpalCt_SetNAD(cmd_buff[0]);

    switch(phStatus)
    {

        case PH_ERR_SUCCESS:

            return ALPAR_ERR_SUCCESS;

        case PH_COMP_PAL_CT | PH_ERR_CT_DADSAD_NOT_SUPPORTED:

            return ALPAR_ERR_BAD_NAD;

        default:

            // TODO: return generic error code
            ;

    }

    return ALPAR_ERR_SUCCESS;

}


phStatus_t alpar_idle_mode(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Idle Mode\n");

    if( (cmd_buff[0] != 0x00 && cmd_buff[0] != 0x01) || (cmd_buff[1] != 0x00 && cmd_buff[1] != 0x01) )
    {
        return ALPAR_ERR_BAD_PARAMETER;
    }

    phStatus_t phStatus = PH_ERR_FAILED;

    phStatus = phhalCt_ClockStartStop( cmd_buff[0], cmd_buff[1]);

    switch(phStatus)
    {

        case PH_ERR_SUCCESS:

            return ALPAR_ERR_SUCCESS;

        case PH_COMP_HAL_CT | PH_ERR_INVALID_PARAMETER:

            return ALPAR_ERR_BAD_PARAMETER;

        default:

            // TODO: return generic error code.
            ;

    }

    return ALPAR_ERR_SUCCESS;

}


phStatus_t alpar_get_reader_status(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Get Reader Status\n");

    if ((PH_ERR_CT_MAIN_CARD_PRESENT | PH_COMP_HAL_CT) == phhalCt_CheckCardPres())
        resp_buff[0] = 0x01;
    else
        resp_buff[0] = 0x00;

    *resp_len = 1;

    return ALPAR_ERR_SUCCESS;

}

phStatus_t alpar_outgoing_card_presence(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    resp_buff[0] = cmd_buff[0];

    *resp_len = 1;

    cardActivated = 0x00;

    return ALPAR_ERR_SUCCESS;

}

phStatus_t alpar_processActivation(phpalCt_DataParams_t *pDataParams,
        phhalCt_ProtocolParams_t *psAtrPalParams)
{
    phStatus_t eStatus = PH_ERR_SUCCESS;
    if(psAtrPalParams->bNegotiableMode)
    {
        phhalCt_PPSParamType_t sPPSParam;
        sPPSParam.bBWI = psAtrPalParams->bBWI;
        sPPSParam.bCWI = psAtrPalParams->bCWI;
        sPPSParam.bWI = psAtrPalParams->bWI;
        sPPSParam.bProtSelT0 = psAtrPalParams->bProtSelT0;
        sPPSParam.bProtSelT1 = psAtrPalParams->bProtSelT1;

        if(!(psAtrPalParams->bProtSelT0))
        {
            sPPSParam.bNegotiableProtocol = E_PHPAL_CT_T1;
        }
        else
        {
            sPPSParam.bNegotiableProtocol = E_PHPAL_CT_T0 ;
        }
        sPPSParam.bFiDi  = psAtrPalParams->bFiDi;

         /* Delay of 10 Etu is required for BGT.*/
         phUser_Wait(1000);
         (void)phhalCt_SetTimer(PHHAL_CT_PPSMODE_9600_ETU,0x00);
         eStatus = phhalCt_PPSRequestHandling( &sPPSParam);
         PHHAL_CT_STOP_TIMER;
         if (PH_ERR_SUCCESS != eStatus)
         {
             if( (PH_COMP_HAL_CT | PH_ERR_PPS_EXCHANGE_NOT_REQUIRED) == eStatus)
             {
                 eStatus = PH_ERR_SUCCESS;
             }
             else
             {
                 /* Call deactivation  */
                 (void) phhalCt_CardDeactivate();
                 return eStatus;
             }
         }
         psAtrPalParams->bProtSelT0 = sPPSParam.bProtSelT0;
         psAtrPalParams->bProtSelT1 = sPPSParam.bProtSelT1;

         bWI = sPPSParam.bWI;
         bBWI = sPPSParam.bBWI;
         bCWI = sPPSParam.bCWI;

     }



     gphpalCt_BIFSC = psAtrPalParams->bIFSC;


     if(!(psAtrPalParams->bProtSelT0))
     {
         pDataParams->bCardSupportedProt = E_PHPAL_CT_T1;
         gphpalCt_ProtSel = E_PHPAL_CT_T1;
         phpalCt_T1_Init();
     }
     else if((psAtrPalParams->bProtSelT0)&& (psAtrPalParams->bProtSelT1))
     {
         pDataParams->bCardSupportedProt = E_PHPAL_CT_BOTH_T0_T1;
         gphpalCt_ProtSel = E_PHPAL_CT_BOTH_T0_T1;
     }
     else
     {
         pDataParams->bCardSupportedProt = E_PHPAL_CT_T0;
         gphpalCt_ProtSel = E_PHPAL_CT_T0;
     }

     /* Wait atleast 10 etu for not violating BGT */
     phUser_Wait(1000);
     return eStatus;
}

phStatus_t alpar_PowerUp(uint8_t bVccSel, uint8_t bEmvEn, uint8_t* pbAtrBuffer, uint8_t* pbAtrSize)
{
    phpalCt_DataParams_t pDataParams;

    pDataParams.pbAtrBuffer = pbAtrBuffer;
    pDataParams.bSizeOfATRbuffer = PH_EXCTEMVCO_MAX_ATR_SIZE;

    phStatus_t eStatus = PH_ERR_INVALID_PARAMETER;
    phhalCt_ProtocolParams_t sAtrPalParams;

    negotiableMode = 0x00;

    if( bEmvEn == 0x00 )
    {
        gphhalCt_BEmvEn = 0x00;
    }
    else if ( bEmvEn == 0x01 )
    {
        gphhalCt_BEmvEn = 0x01;
    }
    else
    {
        return ALPAR_ERR_BAD_PARAMETER;
    }

    eStatus = phhalCt_CardActivate(bVccSel, pDataParams.pbAtrBuffer,
                                     &(pDataParams.bAtrReceivedLength),&sAtrPalParams);

    if(eStatus == ( PH_COMP_HAL_CT | PH_ERR_CT_CARD_ALREADY_ACTIVATED ) )
    {
        eStatus = phhalCt_WarmReset(pDataParams.pbAtrBuffer, &(pDataParams.bAtrReceivedLength),&sAtrPalParams);
    }

    bWI = sAtrPalParams.bWI;
    bBWI = sAtrPalParams.bBWI;
    bCWI = sAtrPalParams.bCWI;

    if (eStatus == PH_ERR_SUCCESS)
    {
        eStatus = alpar_processActivation(&pDataParams, &sAtrPalParams);

        switch(eStatus)
        {

            case PH_ERR_SUCCESS:
            case PH_COMP_HAL_CT | PH_ERR_PPS_EXCHANGE_NOT_REQUIRED:

                pDataParams.bSupportedClass = bVccSel;

                break;

            case PH_COMP_HAL_CT | PH_ERR_INVALID_PARAMETER:

                return ALPAR_ERR_BAD_PARAMETER;

            case PH_COMP_HAL_CT | PH_ERR_CT_PPS_EXCHANGE_ERROR:

                return ALPAR_ERR_PPS_BAD_ANSWER;

            default:

                // TODO: Return generic error code.
                ;

        }

    }
    else
    {
        switch(eStatus)
        {

            case PH_ERR_OPERATION_TIMEDOUT:
                //Api timed out.

                return ALPAR_ERR_EXCEEDED_ATR_DURATION;

            case PH_COMP_HAL_CT | PH_ERR_INVALID_PARAMETER:
                //Parameters are invalid.

                return ALPAR_ERR_BAD_PARAMETER;

            case PH_COMP_HAL_CT | PH_ERR_CT_MAIN_CARD_ABSENT:
                //Card is absent in the slot.

                return ALPAR_ERR_CARD_ABSENT;

            case PH_COMP_HAL_CT | PH_ERR_CT_CARD_ALREADY_ACTIVATED:
                //Card is already activated.

                // TODO: Warm reset

                break;

            case PH_COMP_HAL_CT | PH_ERR_CT_MUTE_ERROR:
                //Card is muted.

                return ALPAR_ERR_CARD_MUTE_PWR_ON;

            case PH_COMP_HAL_CT | PH_ERR_CT_EARLY_ERROR:
                //Card has answered early.

                return ALPAR_ERR_EARLY_ANSWER;

            case PH_COMP_HAL_CT | PH_ERR_CT_PARITY_ERROR:
                //Card has parity error while receiving the ATR.

                return ALPAR_ERR_ATR_PARITY_ERR;

            case PH_COMP_HAL_CT | PH_ERR_CT_OVERUN_ERROR:
                //Fifo is over run while receiving the ATR.

                return ALPAR_ERR_CARD_OVERFLOW;

            case PH_COMP_HAL_CT | PH_ERR_CT_FRAMING_ERROR:
                //Framing error while receiving the ATR.

                return ALPAR_ERR_FRAMING_ERR;

            case PH_COMP_HAL_CT | PH_ERR_CT_TEMPERATURE_LATCHED:
                //This error bit is set when a temperature latch occurs.

                return ALPAR_ERR_TEMPERATURE_ALARM;

            case PH_COMP_HAL_CT | PH_ERR_CT_PROTECTION_LATCHED:
                //This error bit is set when a protection is latched.

                return ALPAR_ERR_SPLY_VOLTAGE_DROPOFF;

            case PH_COMP_HAL_CT | PH_ERR_CT_CLASS_CHANGE_INDICATED:
                //When prootocol 15 is present with its TA value in 7816.

                // TODO: Return error to indicate class change indicated.

                break;

            case PH_COMP_HAL_CT | PH_ERR_CT_ATR_SPECIFIC_PARAMETER_UNSUPPORTED:
                //In 7816 profile,in Specific mode either baud rate or protocol is unsupported.

                return ALPAR_ERR_BAD_PARAMETER;

            case PH_COMP_HAL_CT | PH_ERR_CT_ATR_PARSER_ERROR:
                //ATR parser failed, ATR is inconsistent with the specification.

                return ALPAR_ERR_ATR_NOT_SUPPORTED;

            default:

                // TODO: Return generic error code.
                ;

        }
    }

    negotiableMode = sAtrPalParams.bNegotiableMode;
    *pbAtrSize = (uint16_t) pDataParams.bAtrReceivedLength;

    cardActivated = 0x01;

    return ALPAR_ERR_SUCCESS;
}

phStatus_t alpar_power_up_5V(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Power Up 5V\n");

    phStatus_t phStatus = PH_ERR_FAILED;

    phStatus = alpar_PowerUp(PHHAL_CT_VCC5, cmd_buff[0], resp_buff, (uint8_t *) resp_len);

    return phStatus;

}


phStatus_t alpar_power_up_3V(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Power up 3V\n");

    phStatus_t phStatus = PH_ERR_FAILED;

    phStatus = alpar_PowerUp(PHHAL_CT_VCC3, cmd_buff[0], resp_buff, (uint8_t *) resp_len);

    return phStatus;

}

phStatus_t alpar_power_up_1V8(uint8_t *cmd_buff, uint16_t *cmd_len, uint8_t *resp_buff, uint16_t *resp_len)
{

    LOG_TXT("Power up 1V8\n");

    phStatus_t phStatus = PH_ERR_FAILED;

    phStatus = alpar_PowerUp(PHHAL_CT_VCC1M8, cmd_buff[0], resp_buff, (uint8_t *) resp_len);

    return phStatus;

}
