/**
 * Copyright 2021 NXP
 *
 * This software is owned or controlled by NXP and may only be used
 * strictly in accordance with the applicable license terms.  By expressly
 * accepting such terms or by downloading, installing, activating and/or
 * otherwise using the software, you are agreeing that you have read, and
 * that you agree to comply with and are bound by, such license terms.  If
 * you do not agree to be bound by the applicable license terms, then you
 * may not retain, install, activate or otherwise use the software.
 *
 */

/** @file
 *
 * phExCTEMVCO.c:
 *
 *
 * Project:  PN7462AU
 *
 * $Date: 2021-09-23 15:00:00 +0200 (Thu, 23 Sep 2021) $
 * $Author: Petar Hlad (MobileKnowledge) $
 * $Revision:
 */


/* *****************************************************************************************************************
 * Includes
 * ***************************************************************************************************************** */

#include <stdio.h>
#include "ph_Datatypes.h"
#include "phFlashBoot_Event.h"
#include "phCfg_EE.h"
#include "phUser.h"
#include "phFlashBoot.h"
#include "phpalCt.h"
#include "phhalCt.h"
#include "phExCTEMVCo.h"
#include "phLED.h"
#include "ph_Log.h"
#include "phhalTimer.h"
#include "PN7462AU/PN7462AU_hif.h"
#include "phhalHif.h"
#include "phhalPcr.h"
#include "alpar.h"
#include "phRtos.h"


/* *****************************************************************************************************************
 * Type Definitions
 * ***************************************************************************************************************** */

/* *****************************************************************************************************************
 * Global and Static Variables
 * Total Size: NNNbytes
 * ***************************************************************************************************************** */
static uint8_t bApduSelectPPSE[] =
{ 0x00, 0xA4, 0x04, 0x00, 0x0E, 0x31, 0x50, 0x41, 0x59, 0x2E, 0x53, 0x59, 0x53, 0x2E, 0x44, 0x44, 0x46, 0x30, 0x31, 0x00 };


PH_NOINIT static phhalTimer_Timers_t *pLedTimer;

void phEMVCo_LoopbackProcess(void);



static phhalTimer_Timers_t *pLedTimer;

static phhalHif_Hsu_BaudRate_t bHsuHifBaudRate;
phhalHif_Config_t gphEx_HsuHifConfig;

uint32_t gphEx_SysHsuTxBuffer[PH_EX_HSU_BUFFER_SIZE];
uint8_t gphEx_SysHsuRxBuffer[PH_EX_HSU_BUFFER_SIZE];

uint8_t txBuffer[4*PH_EX_HSU_BUFFER_SIZE];


volatile uint8_t gphUART_hsuReceiveComplete;

static phStatus_t phExHif_GetRxDataLength(phhalHif_BuffId_t bBuffId, uint16_t *wLength);

extern void HIF_IRQHandler(void);

/* *****************************************************************************************************************
 * Private Functions Prototypes
 * ***************************************************************************************************************** */
void phEx_HsuHifErrorCallBack(uint32_t dwIntStatus);
void phEx_Fill_Tx_Buffer(uint8_t* buff, uint16_t len );
void phEx_HsuHifSendBuff(uint8_t* buffin, uint16_t len);
void phEx_HsuHifConfig();
void phEx_Hsu_Print(uint32_t *inBuf, uint16_t len);
void phEx_HsuHifTransmitCallBack(uint32_t dwIntStatus);
void phEx_HsuHif_ReceiveCallBack(uint32_t dwIntStatus);

phStatus_t phEMVLoopbackCheckCardStatus(uint8_t * bCardStatus);
phStatus_t phEMVLoopbackCardActivation(uint8_t * gbphCTEMVCo_ATRBuffer, uint16_t * bAtrSize);
phStatus_t phEMVLoopbackTransceiveAPDU(uint8_t * bAPDUcmd, uint32_t bCmdSize, uint8_t * bAPDUResponse, uint16_t * bRespSize);

uint32_t phEx_HsuHif_Read(uint8_t *pBuf, uint16_t* buf_len);

uint8_t compute_lrc(uint8_t *array, uint16_t len);

void tx_response(uint8_t ack, uint8_t cmd, uint8_t *data, uint16_t data_length);

void tx_response(uint8_t ack, uint8_t cmd, uint8_t *data, uint16_t data_length){
    txBuffer[0] = ack ? 0x60 : 0xE0;

    txBuffer[1] = (uint8_t) data_length >> 8;
    txBuffer[2] = (uint8_t) data_length & 0x00FF;

    txBuffer[3] = cmd;

    if(data_length > 0)
    {
        memcpy(&txBuffer[4], data, data_length);
    }

    txBuffer[4 + data_length] = compute_lrc(txBuffer, 4 + data_length);

    LOG_AU8("ALPAR tx:\n",txBuffer, 5 + data_length);

    phEx_HsuHifSendBuff(txBuffer, 5 + data_length);

}

uint8_t compute_lrc(uint8_t *array, uint16_t len)
{
    uint8_t lrc = 0;
    uint16_t i = 0;
    for(;i<len;++i)
    {
        lrc ^= array[i];
    }

    return lrc;
}

/* *****************************************************************************************************************
 * Public Functions
 * ***************************************************************************************************************** */

static void phExMain_CtCallBack(uint32_t dwIRQMessage)
{

    phStatus_t status = 0x00;
    uint8_t presence[1];


    if(dwIRQMessage & CT_USR2_REG_PRESL_MASK)
    {
        status = phhalCt_CheckCardPres();
    }

    if(status == (PH_ERR_CT_MAIN_CARD_PRESENT | PH_COMP_HAL_CT) )
    {
        LOG_TXT("Card inserted\n");
        presence[0] = 0x01;
    }
    else if(status == (PH_ERR_CT_MAIN_CARD_ABSENT | PH_COMP_HAL_CT))
    {
        LOG_TXT("Card Extracted\n");
        presence[0] = 0x00;
    }
    else
    {
        return;
    }

    alpar_handle_command( ALPAR_OUTGOING_CMD_CARD_PRES, &presence[0], 1);

}

/* *****************************************************************************************************************
 * Private Functions
 * ***************************************************************************************************************** */
/** phExCTEMVCo entry point. */
int main(void)
{

	uint16_t bHSUlen = 0;

	uint32_t dwIRQMessage = CT_USR2_REG_PRESL_MASK;

    phhalCt_InitParam_t  sInitParams;
    sInitParams.bCardPresConnectorStatus= gpkphCfg_EE_Boot_CT->bConnectorType;
    sInitParams.bPullupSetStatus = gpkphCfg_EE_Boot_CT->bPullUp;
    sInitParams.bSlewRateConfig = gpkphCfg_EE_Boot_CT->bSlewRate;
    sInitParams.bAutoCTDeactivationEnable= 0;

    phFlashBoot_Main();

    /** Initialize delay loop variable based on CPU speed */
    phUser_WaitInit((phUser_CpuSpeed_t)E_COMMON_CPUSPEED_20MHZ);

    PH_USER_ASSERT(phhalTimer_RequestTimer(E_TUNIT_MILLI_SECS,  &pLedTimer) == PH_ERR_SUCCESS);

    /** Initialize the LED module */
    phLED_Init();

    phLED_SetPattern(gkphLED_BootUp);
    phhalTimer_Configure(pLedTimer, 100 , &phLED_TimerCallback);
    phhalTimer_Start(pLedTimer, E_TIMER_FREE_RUNNING);

    phpalCt_Init(& sInitParams);

    bHsuHifBaudRate = E_HSU_BAUDRATE_115_2K;
	phEx_HsuHifConfig();

    /* Register the call back */
    phhalCt_RegCallBack(&phExMain_CtCallBack,dwIRQMessage);


	do
	{
	    //blocking call
	    phEx_HsuHif_Read(gphEx_SysHsuRxBuffer, &bHSUlen);

	    LOG_AU8("ALPAR rx:\n",gphEx_SysHsuRxBuffer, bHSUlen);

        if ( (bHSUlen < 5) || (gphEx_SysHsuRxBuffer[0] != 0x60) )
        {

            tx_response(0, ALPAR_ERR_UNKNOWN_COMMAND, 0, 0);

        }
        else if ( compute_lrc(gphEx_SysHsuRxBuffer, bHSUlen) != 0x00 )
        {
            /* Test LRC check ok */

            //LRC Error
            LOG_TXT("Wrong LRC\n");

            uint8_t alpar_err = ALPAR_ERR_SERIAL_LCR_ERR;
            tx_response(0, gphEx_SysHsuRxBuffer[3], &alpar_err, 1);

        }
        else {

            alpar_handle_command(gphEx_SysHsuRxBuffer[3], &gphEx_SysHsuRxBuffer[4], bHSUlen - 5);

        }

	} while(1);

    return 0;

}

/**
 * phEMVLoopbackCardActivation
 */
phStatus_t phEMVLoopbackCardActivation(uint8_t * gbphCTEMVCo_ATRBuffer, uint16_t * bAtrSize)
{
    phpalCt_DataParams_t pDataParams ;
    phStatus_t phStatus = PH_ERR_FAILED;

    LOG_TXT("Card Activation\n");

    /* The response buffer is used to optimize the code */
    pDataParams.pbAtrBuffer = (uint8_t *)gbphCTEMVCo_ATRBuffer;
    pDataParams.bSizeOfATRbuffer = PH_EXCTEMVCO_MAX_ATR_SIZE;

    phStatus = phpalCt_ActivateCard(&pDataParams );
    if((PH_ERR_SUCCESS != phStatus))
    {

        if((PH_ERR_CT_EARLY_ERROR| PH_COMP_HAL_CT) == phStatus)
        {
            LOG_TXT("Early Atr is received..\n");
            return phStatus;
        }
        else if((PH_ERR_CT_ATR_PARSER_ERROR|PH_COMP_HAL_CT)== phStatus)
        {
            phLED_SetPattern(gkphLED_Ct_Fail);
            if(pDataParams.bAtrReceivedLength <= 32)
            {
                LOG_AU8("ATR:\n",gbphCTEMVCo_ATRBuffer, pDataParams.bAtrReceivedLength);
            }
            LOG_TXT("Non EMVCo Card detected.\n");
            return phStatus;
        }
        else
        {
            phLED_SetPattern(gkphLED_Ct_Fail);
            LOG_TXT("Card Activation failed.\n");
            LOG_X32("Error Status", phStatus);
            return phStatus;
        }
    }
    LOG_AU8("ATR:\n",pDataParams.pbAtrBuffer, pDataParams.bAtrReceivedLength);
    LOG_TXT("\nEMVCo payment card detected. Checking embedded VISA or Master Card applications..\n");

    *bAtrSize = pDataParams.bAtrReceivedLength;

    return phStatus;
}



/**
 * phEMVLoopbackCheckCardStatus
 */
phStatus_t phEMVLoopbackCheckCardStatus(uint8_t * bCardStatus)
{

	LOG_TXT("Check Presence\n");

    if ((PH_ERR_CT_MAIN_CARD_PRESENT | PH_COMP_HAL_CT) == phhalCt_CheckCardPres())
    	*bCardStatus = 0x01;
    else
    	*bCardStatus = 0x00;

    return PH_ERR_SUCCESS;

}


/**
 * phEMVLoopbackTransceiveAPDU
 */
phStatus_t phEMVLoopbackTransceiveAPDU(uint8_t * bAPDUcmd, uint32_t bCmdSize, uint8_t * bAPDUResponse, uint16_t * bRespSize)
{
	phStatus_t phStatus = PH_ERR_FAILED;

	LOG_AU8("ALPAR Send:\n",bAPDUcmd, bCmdSize);

    /** Send the Apdu */
    phStatus = phpalCt_Transceive(
    		bAPDUcmd,
			bCmdSize,
			bAPDUResponse, bRespSize);

    if (phStatus != PH_ERR_SUCCESS)
    {
    	*bRespSize = 0;
    	return phStatus;
    }

    LOG_AU8("ALPAR Receive:\n",bAPDUResponse, *bRespSize);

    return PH_ERR_SUCCESS;
}




/**
 * phExCTEMVCo_ALPAREmuLoopbackProcess
 *
 * Test APIs for ALPAR Emu
 */

// For EMV L1 Protocol debugging it is practical to avoid to have to extract and insert the card on each test
#define EMV_FAST_TESTING 1

void phEMVCo_LoopbackProcess(void)
{
	uint8_t bEndOfLoopBack, bCardPresent = 0;
	uint8_t bATRBuf[64];
	uint8_t bRespBuf[270];
	uint8_t bCmdBuf[270];
	uint16_t i, bRspSz;
	uint32_t bCmdSz;
	uint16_t bATRSize = 0;
	uint8_t successive_APDU_err = 0;
	phStatus_t phStatus = PH_ERR_FAILED;

    do
    {
        LOG_TXT("Waiting for CT card insertion.\n");
        /** Check the card presence */
        while(!bCardPresent)
        {
        	phEMVLoopbackCheckCardStatus(&bCardPresent);
        }

        LOG_TXT("CT card detected.\n");

#if EMV_FAST_TESTING == 0
        phRtos_TaskDelay(10000);
#else
        phRtos_TaskDelay(2000);
#endif

        LOG_TXT("Activating CT Card.\n");

        phStatus = phEMVLoopbackCardActivation(bATRBuf, &bATRSize);

        if((PH_ERR_SUCCESS != phStatus))
        {

            phhalCt_CardDeactivate();

#if EMV_FAST_TESTING == 0
            while(bCardPresent)
            {
                phEMVLoopbackCheckCardStatus(&bCardPresent);
            }
#endif

            continue;
        }

        LOG_AU8("ATR:\n", bATRBuf, bATRSize);
        LOG_TXT("\nEMVCo payment card detected.\n");

        phStatus = phpalCt_Transceive(bApduSelectPPSE, (uint32_t)sizeof(bApduSelectPPSE), bRespBuf, (uint16_t *)&bRspSz);

        bEndOfLoopBack = 0;
        while (0 == bEndOfLoopBack)
        {
            /* Exchange OK ? */
            if(phStatus != PH_ERR_SUCCESS || bRspSz < 6)
            {

                if(phStatus != PH_ERR_SUCCESS) {

                    /**
                     * Exchange error
                     */

                    LOG_TXT("Error transceiving Apdu\n");
                    LOG_X32("Error Status :  ", phStatus);

#if EMV_FAST_TESTING == 1
                    ++successive_APDU_err;

                    if (successive_APDU_err > 0) {
                        bEndOfLoopBack = 1;
                        successive_APDU_err = 0;
                        continue;
                    }
#endif

                }

                phStatus = phEMVLoopbackTransceiveAPDU(bApduSelectPPSE, (uint32_t)sizeof(bApduSelectPPSE), bRespBuf, (uint16_t *)&bRspSz);

                bEndOfLoopBack = 0;

            }
            else
            {

                /* End of loopback ? */
                if (0x70 == bRespBuf[1])
                {
                    /* Second byte = 0x70, stop the loopback */

                    bEndOfLoopBack = 1;

                    //fnUartPrintf("\nINS 70 End of loopback\n");

                }
                else
                {

                    /* Format the card response into a new command without the status word 0x90 0x00 */
                    for(i = 0; i < (bRspSz - 2); i++)
                    {
                        bCmdBuf[i] = bRespBuf[i];
                    }
                    /* set the command size */
                    bCmdSz = bRspSz-2;

                    /* Send of a new command */
                    /** Send the Apdu */
                    phStatus = phEMVLoopbackTransceiveAPDU(bCmdBuf, bCmdSz, bRespBuf, &bRspSz);

                    bEndOfLoopBack = 0;
                }

            }

            phRtos_TaskDelay(200);

        }


        phhalCt_CardDeactivate();

#if EMV_FAST_TESTING == 0
        while(bCardPresent)
        {
            phEMVLoopbackCheckCardStatus(&bCardPresent);
        }
#endif

    } while(1);

}

void phEx_HsuHifSendBuff(uint8_t* buffin, uint16_t len)
{
    phEx_Fill_Tx_Buffer(buffin, len);
    phEx_Hsu_Print(gphEx_SysHsuTxBuffer,len);
}


void phEx_HsuHifConfigBaudRate(phhalHif_Hsu_BaudRate_t baud_rate)
{
    bHsuHifBaudRate = baud_rate;
    phEx_HsuHifConfig();
}


void phEx_HsuHifConfig(void)
{
    /*Configure HSU interface*/
    gphEx_HsuHifConfig.eInterface = E_HIF_HSU;
    gphEx_HsuHifConfig.sConfig.sHsuConfig.bBaudRate = bHsuHifBaudRate;
    gphEx_HsuHifConfig.sConfig.sHsuConfig.bDummyBytes = 0;

    gphEx_HsuHifConfig.sConfig.sHsuConfig.bStopBits = 1;
    gphEx_HsuHifConfig.sConfig.sHsuConfig.bEOF = 0x8;
    gphEx_HsuHifConfig.bTimeout = 0x00;

    gphEx_HsuHifConfig.eBufferType = E_BUFFER_FORMAT_FREE;
    gphEx_HsuHifConfig.bShortFrameLen = 0;
    gphEx_HsuHifConfig.bStoreErrData = 0;


    gphUART_hsuReceiveComplete = FALSE;

    phhalHif_Init(&gphEx_HsuHifConfig,(pphhalHif_Callback_t)&phEx_HsuHifErrorCallBack);


    /** Configure Receive Buffer0 with 1024 max length for data reception */
   // phStatus_t phhalHif_InitRxBuffer(phhalHif_BuffId_t sBuffId, uint16_t wMaxSize,
   // uint32_t *pdwBuffAddr, pphhalHif_Callback_t pCallBackPtr);

    PH_USER_ASSERT(
            phhalHif_InitRxBuffer(E_RX_BUFFER_ID0, (PH_EX_HSU_BUFFER_SIZE * 4), (uint32_t *)gphEx_SysHsuRxBuffer,\
            		(pphhalHif_Callback_t)&phEx_HsuHif_ReceiveCallBack)== PH_ERR_SUCCESS);

}



uint32_t phEx_HsuHif_Read(uint8_t *pBuf, uint16_t* buf_len)
{
    do {
	phStatus_t eStatus = PH_ERR_SUCCESS;


	while(!gphUART_hsuReceiveComplete);
	gphUART_hsuReceiveComplete = FALSE;

    eStatus = phExHif_GetRxDataLength((phhalHif_BuffId_t) E_RX_BUFFER_ID0,
    		(uint16_t *)buf_len);
    PH_BREAK_ON_FAILURE(eStatus);
    memcpy(pBuf, gphEx_SysHsuRxBuffer, *buf_len);

	eStatus = phhalHif_ReleaseRxBuffer((phhalHif_BuffId_t) E_RX_BUFFER_ID0);
	PH_BREAK_ON_FAILURE(eStatus);

    } while (0);

    return PH_ERR_SUCCESS;
}


void phEx_HsuHifErrorCallBack(uint32_t dwIntStatus)
{
// Add the code which is to be processed during HIF error interrupt.
}


void phEx_Fill_Tx_Buffer(uint8_t* inBuff, uint16_t len)
{
	uint16_t i;
	uint16_t j = 0;
	uint8_t *p;
	p = (uint8_t*)gphEx_SysHsuTxBuffer;
	for (i = 0; i <=  len; i++)
	{
		*(p+i)  = inBuff[j];
		j++;
	}

}

void phEx_Hsu_Print(uint32_t *inBuf,uint16_t len)
{
    /*Print on the PC console using HSU interface through USB to serial converter*/
    phhalHif_Transmit((uint32_t *)inBuf,len,(pphhalHif_Callback_t)&phEx_HsuHifTransmitCallBack);
}



void phEx_HsuHifTransmitCallBack(uint32_t dwIntStatus)
{
    memset(gphEx_SysHsuTxBuffer,0,PH_EX_HSU_BUFFER_SIZE);
}



void phEx_HsuHif_ReceiveCallBack(uint32_t dwIntStatus)
{
	//set event receive eor
    gphUART_hsuReceiveComplete = TRUE;
	//phhalHif_ReleaseRxBuffer(E_RX_BUFFER_ID0);// Release the receive buffer so that PN640 can receive the next packet.
}


/** Get the length of data received*/
static phStatus_t phExHif_GetRxDataLength(phhalHif_BuffId_t bBuffId,
        uint16_t *wLength) {
    *wLength = (uint16_t) PH_REG_GET_FIELD_NS_SC(
            (HOSTIF_BUFFER_RX0_LEN_REG + ((uint32_t)bBuffId*4)),
            HOSTIF_BUFFER_RX0_LEN_REG_RX0_LENGTH_MASK);
#if HOSTIF_BUFFER_RX0_LEN_REG_RX0_LENGTH_POS != 0
#   error The above logic expects that we are reading from the Lower bytes without shifting.
    /* if the reading position from register is not 0, this has to be handled */
#endif
    return PH_ERR_SUCCESS;
}

