/*
 * licenseMgr.c
 *
 *  Created on: 16.03.2010
 *      Author: stefan.detter
 */

#include "licenseMgr.h"
#include "../protocol/licenseProtocol.h"

#include <utility/trace.h>

#include <hal/flags/flags.h>

#include <core/store/store.h>
#include <core/report/report.h>

//------------------------------------------------------------------------------
//  	Variables
//------------------------------------------------------------------------------
#define LICENSE_DATA_FLAG_OK			0x55AA11EE

static LICENSE_MODE g_mode;


typedef struct {
	ulong 	startReaderId;
	ulong 	startLicenseCount;
	ulong 	nextReaderId;
	ulong 	remainingLicenseCount;
} LICENSE_DATA;

static LICENSE_DATA		g_licenseData;


#define SAVE_LICENSE_DATA()			store_writeData(LICENSE_DATA_ADDRESS + 4, (uchar*)&g_licenseData, sizeof(LICENSE_DATA))
#define LOAD_LICENSE_DATA()			store_readData(LICENSE_DATA_ADDRESS + 4, (uchar*)&g_licenseData, sizeof(LICENSE_DATA))

//------------------------------------------------------------------------------
//  	Global Functions
//------------------------------------------------------------------------------

void licmgr_init()
{
	ulong flag = 0;

#if DBG_LICMGR
	TRACE_INFO("LicMgr:\t Init\n\r");
#endif

	store_readData(LICENSE_DATA_ADDRESS, (uchar*)&flag, 4);
	if(flag != LICENSE_DATA_FLAG_OK)
	{
#if DBG_LICMGR
		TRACE_INFO("LicMgr:\t OK-Flag not present, initializing data and writing OK-Flag\n\r");
#endif

		g_licenseData.startLicenseCount = START_LICENSE_COUNT;
		g_licenseData.remainingLicenseCount = START_LICENSE_COUNT;
		g_licenseData.startReaderId = START_READER_ID;
		g_licenseData.nextReaderId = START_READER_ID;

		SAVE_LICENSE_DATA();

		flag = LICENSE_DATA_FLAG_OK;
		store_writeData(LICENSE_DATA_ADDRESS, (uchar*)&flag, sizeof(flag));
	}

#if DBG_LICMGR
	TRACE_INFO("LicMgr:\t OK-Flag present, reading data\n\r");
#endif
	LOAD_LICENSE_DATA();


#if DBG_LICMGR
	TRACE_INFO("LicMgr:\t Current LicenseData:\n\r");
	TRACE_INFO("LicMgr:\t \tStart   ReaderId:       %08X\n\r", (uint)g_licenseData.startReaderId);
	TRACE_INFO("LicMgr:\t \tStart   LicenseCount:   %d\n\r", (uint)g_licenseData.startLicenseCount);
	TRACE_INFO("LicMgr:\t \tCurrent ReaderId:       %08X\n\r", (uint)g_licenseData.nextReaderId);
	TRACE_INFO("LicMgr:\t \tCurrent LicenseCount:   %d\n\r", (uint)g_licenseData.remainingLicenseCount);
#endif
}

/*!
 * @brief sets the LicenseMode: READERID_NEEDED | READERID_AVAILABLE
 *
 * @param m the idStore byte out of the payload buffer
 */
bool licmgr_setLicenseMode(uchar mode)
{
	// check idStore Flag
	if( mode == (uchar)READERID_AVAILABLE )
	{
#if DBG_LICMGR
		TRACE_INFO("LicMgr:\t +++ ID AVAILABLE +++\n\r");
#endif
		g_mode = READERID_AVAILABLE;
	}
	else if( mode == (uchar)READERID_NEEDED )
	{
#if DBG_LICMGR
		TRACE_INFO("LicMgr:\t +++ ID NEEDED +++\n\r");
#endif
		g_mode = READERID_NEEDED;
	}
	else
	{
		TRACE_FATAL("LicMgr:\t --- Wrong License Mode -> Abort! ---\n\r");
		return false;
	}
	return true;
}

LICENSE_MODE licmgr_licenseMode()
{
	return g_mode;
}

/*!
 * @brief checks if there are anymore licenses available
 *
 * @return true if there still are licenses left,
 * 		   otherwise false
 */
bool licmgr_isLicenseAvailable ( )
{
	if(g_licenseData.remainingLicenseCount > 0)
		return true;
	return false;
}

/*!
 * @brief Stores the next available reader id in "readerId"
 *
 * @param readerType contains the 4 Byte reader Type
 * @param readerId 	 the actual reader id will be stored here ( 4 Byte )
 *
 * @return True if license is available
 * 		   False if no license is available
 */
bool licmgr_getNextReaderId ( ulong* readerId )
{
	// select license mode
	if(g_mode == READERID_AVAILABLE)
	{
		*readerId = (ulong) ID_MAGIC_READERID;
	}
	else if(g_mode == READERID_NEEDED)
	{
		*readerId = g_licenseData.nextReaderId;
	}
	else
	{
		return false;
	}

#if DBG_LICMGR
	TRACE_INFO("LicMgr:\t Next ReaderId requested: %08X\n\r", (uint)*readerId);
#endif

	return true;
}


void licmgr_storeData ( )
{
#if DBG_LICMGR
	TRACE_INFO("LicMgr:\t Saving data with mode -> %s\n\r", (g_mode == READERID_NEEDED)?"ID Needed" : "ID Available");
#endif

	// if is needed -> standard
	if( g_mode == READERID_NEEDED )
	{
		g_licenseData.nextReaderId++;
	}

	g_licenseData.remainingLicenseCount--;

	SAVE_LICENSE_DATA();

#if DBG_LICMGR
	TRACE_INFO("LicMgr:\t Current LicenseData:\n\r");
	TRACE_INFO("LicMgr:\t \tStart     ReaderId:       %08X\n\r", (uint)g_licenseData.startReaderId);
	TRACE_INFO("LicMgr:\t \tStart     LicenseCount:   %d\n\r", (uint)g_licenseData.startLicenseCount);
	TRACE_INFO("LicMgr:\t \tNext      ReaderId:       %08X\n\r", (uint)g_licenseData.nextReaderId);
	TRACE_INFO("LicMgr:\t \tRemaining LicenseCount:   %d\n\r", (uint)g_licenseData.remainingLicenseCount);
#endif
}


void licmgr_getLicenseInfo ( ulong* startReaderId, ulong* startLicenseCount, ulong* nextReaderId, ulong* remainingLicenseCount )
{
	LOAD_LICENSE_DATA();

	*startReaderId = g_licenseData.startReaderId;
	*startLicenseCount = g_licenseData.startLicenseCount;
	*nextReaderId = g_licenseData.nextReaderId;
	*remainingLicenseCount = g_licenseData.remainingLicenseCount;
}


