/*
 * licenseProtocol.c
 *
 *  Created on: 16.03.2010
 *      Author: stefan.detter
 */

#include <core/crypt/aes256.h>
#include <core/random/random.h>
#include <core/licenseMgr/licenseMgr.h>
#include <core/report/report.h>

#include <hal/flags/flags.h>
#include <core/store/store.h>

#include "licenseProtocol.h"
#include "readerHostProtocolDefines.h"
#include "readerHostProtocol.h"
#include "readerHostProtocolImpl.h"

#include <def.h>
#include <utility/trace.h>

//------------------------------------------------------------------------------
//  	"local" Variables
//------------------------------------------------------------------------------

uchar  g_stateStep = 0;

/*! @brief  */
eSTATE g_state = CONNECTING;
/*! @brief  */
eSTATE g_stateBeforeRandom = CONNECTING;

ulong  g_packageCounter = 0;

/*! @brief 256 Bit  AES EBC KEY */
/** @brief 256 Bit  AES EBC KEY */

#if (defined CODING_idt)
uchar g_aesInitialKey[32] = 	{
							0x84, 0x48, 0x24, 0x96, 	0xba, 0x63, 0xb2, 0xc2,
							0x7d, 0x9b, 0x83, 0xa8, 	0xe4, 0x6e, 0x9d, 0x23,
							0xd3, 0xcc, 0x23, 0xa1, 	0xba, 0x46, 0x3a, 0x7d,
							0x45, 0xb3, 0xd1, 0x62, 	0xfd, 0xf3, 0x9b, 0xba,
						};
#endif

/* saves current used AES Key */
uchar  currentKey[AES_KEY_LENGTH];

/*! @brief AES Context saves the current used key */
aes256_context g_aesContext;

//------------------------------------------------------------------------------
//  	Global Variables
//------------------------------------------------------------------------------

/*! @brief byte array of transmitted packet */
uchar  g_sendBuff[20];
/*! @brief char pointer to the payloadstart from g_sendBuff */
uchar* g_sendPayload = &(g_sendBuff[3]);

/*! @brief byte array of the received packet */
uchar  g_recvPayload[16];

uchar  g_exchangeIDPackageCount = 0;

uchar  g_exchangeBlockNr;

uchar  g_randomRunCounter = 0;

ulong  g_BL_readerType = 0;

ulong  g_used_readerId = 0;


extern uint g_timeoutCounter;

//------------------------------------------------------------------------------
//  	Makros
//------------------------------------------------------------------------------

#define TRACE_INFO_ARRAY(array,len) \
{ \
	ushort i = 0; \
	for (i = 0; i < len; i++) \
	{ \
		TRACE_INFO_WP("%02x", array[i]); \
		if (i < len - 1)TRACE_INFO_WP("-"); \
	} \
	TRACE_INFO_WP("\n\r"); \
}

static void SEND_PACKAGE();
static void RESET_STATE_MACHINE();
static void SWITCH_TO_RANDOM_START();
static void fillRequestWithMagicRandVal(uchar magic);
static bool checkResponseWithMagicRandVal(uchar index, uchar magic);
static void fillAESdynamicKeyWithRandom(uchar* key);
static void fillRequestWithRandom();
static void fillRequestWithgetReaderTyp();
static void fillRequestWithEncryptedReaderID(ulong* rid );
static void prepareMessage(uchar* buff);
static bool parseLicenseMessage(uchar* payloadIn, uint payloadIn_length);
static ulong getReaderType ( uchar* buff );

//------------------------------------------------------------------------------
//  	Local Funtions
//------------------------------------------------------------------------------

static void SEND_PACKAGE()
{
	prepareMessage(g_sendBuff);
	sendPackageToHost(g_sendBuff, LICENSE_PACKET_SIZE, RFE_COM1_LICENSE, RFE_COM2_PASS_THROUGH);
#if DBG_LICENSE > 2
	TRACE_INFO("License:\t S> Crypt [%5d]: ", (uint)g_packageCounter++);
	TRACE_INFO_ARRAY(g_sendPayload, 16);
	TRACE_INFO("\n\r");
#endif
}

static void RESET_STATE_MACHINE()
{
	g_state = CONNECTING;
	g_stateStep = 0;
}

static void SWITCH_TO_RANDOM_START()
{
	fillRequestWithRandom();
	g_randomRunCounter = rand_getSpecificRandom(1,32);
	g_sendPayload[RANDOM_INDEX_3] = g_randomRunCounter;
#if DBG_LICENSE
	TRACE_INFO("License:\t  Starting Random procedure with value %02X\n\r", g_randomRunCounter);
#endif
	SEND_PACKAGE();
	g_stateBeforeRandom = g_state;
	g_state = RANDOM_START;
	g_stateStep = 0;
}

static void fillRequestWithMagicRandVal(uchar magic)
{
	fillRequestWithRandom();
	uchar randVal = rand_getSpecificRandom(1,7);
	g_sendPayload[RANDOM_INDEX_1] = randVal;
	g_sendPayload[RANDOM_INDEX_1 + randVal] = (uchar) ((magic) - (randVal << RANDVAL_SHIFT_VALUE));
}

static bool checkResponseWithMagicRandVal(uchar index, uchar magic)
{
	uchar randVal = g_recvPayload[index];
	if (randVal < 1 || randVal > 13 || g_recvPayload[index - randVal] != (magic) + (randVal << RANDVAL_SHIFT_VALUE))
	{
		TRACE_ERROR("License:\t  Wrong Validation values! randVal = %02X | %02X != %02X\n\r", randVal, g_recvPayload[index - randVal], (magic) + (randVal << RANDVAL_SHIFT_VALUE));
		RESET_STATE_MACHINE();
		report_incrementErrorCount(MagicRandValError);
		return false;
	}

	return true;
}


/*! @brief build a new AES key with random values and save it in g_aesDynamicKey */
static void fillAESdynamicKeyWithRandom(uchar* key)
{
	uchar i = 0;
	for (i = 0; i < AES_KEY_LENGTH; i++)
	{
		key[i] = rand_getNextRandomValue() % 256;
	}
}

/*! @brief overwrite the sending packet with random values */
static void fillRequestWithRandom()
{
	uchar i = 0;
	for (i = 0; i < LICENSE_PAYLOAD_SIZE; i++)
	{
#if RANDVALS_ON
		g_sendPayload[i] = rand_getNextRandomValue() % 256;
#else
		g_sendPayload[i] = 0x00;
#endif
	}
}

/*! @brief overwrite the sending packet with getReaderTyp order */
static void fillRequestWithgetReaderTyp()
{
	fillRequestWithRandom();
	g_sendPayload[3]  = CONNECTING_START_BYTE_1;
	g_sendPayload[5]  = CONNECTING_START_BYTE_2;
	g_sendPayload[7]  = CONNECTING_START_BYTE_3;
	g_sendPayload[9]  = CONNECTING_START_BYTE_4;
	g_sendPayload[11] = CONNECTING_START_BYTE_5;
}

/*!
 * @brief Encrypts rid with the XOR_CRYPT values and stores it in buffer at position READER_ID_START_INDEX
 *
 * @param buffer Storage for the encrypted reader id + the checksum
 * @param rid Pointer to the reader ID *
 */
static void fillRequestWithEncryptedReaderID(ulong* rid )
{
	g_sendPayload[READER_ID_START_INDEX + 0] = XOR_CRYPT_READERID_VALUE_1 ^ ((uchar*)(rid))[3];
	g_sendPayload[READER_ID_START_INDEX + 1] = XOR_CRYPT_READERID_VALUE_2 ^ ((uchar*)(rid))[2];
	g_sendPayload[READER_ID_START_INDEX + 2] = XOR_CRYPT_READERID_VALUE_3 ^ ((uchar*)(rid))[1];
	g_sendPayload[READER_ID_START_INDEX + 3] = XOR_CRYPT_READERID_VALUE_4 ^ ((uchar*)(rid))[0];
	g_sendPayload[READER_ID_START_INDEX + 4] = ((uchar*)(rid))[0] ^
										((uchar*)(rid))[1] ^
										((uchar*)(rid))[2] ^
										((uchar*)(rid))[3]; // checksum
}

/*! @brief encryted g_request and send it to reader */
static void prepareMessage(uchar* buff)
{
	uint i;

	buff[0] = RFE_START_BYTE_1;		// Start R
	buff[1] = RFE_START_BYTE_2;		// Start F
	buff[2] = RFE_START_BYTE_3;		// Start E
	buff[19]= 0x00;					//Checksum

	// encrypt data
#if ENCRYPTION_ON

#if DBG_LICENSE > 2
	TRACE_INFO("License:\t S> Plain [%5d]: ", (uint)g_packageCounter);
	TRACE_INFO_ARRAY(g_sendPayload, 16);
#endif

	aes256_encrypt_ecb(&g_aesContext, &(buff[3]));
#endif

	// compute checksum
	for(i = 0; i < 19; i++)
	{
		buff[19] = buff[19] ^ buff[i];
	}
}

/*!
 * @brief License Message Parser
 * after this function, g_response will contain the license message data
 */
static bool parseLicenseMessage(uchar* payloadIn, uint payloadIn_length)
{
	uint i = 0, j = 0;
	static ePROTOCOL licenseProtocolState = RES_START1;

	// parse message
	for(i = 0; i < payloadIn_length; i++)
	{
		//TRACE_INFO("License:\t %3d -> %02x\n\r", licenseProtocolState, payloadIn[i]);
		switch(licenseProtocolState)
		{
		case RES_START1:
			if(payloadIn[i] == RFE_START_BYTE_1)
				licenseProtocolState = RES_START2;
			else
				licenseProtocolState = RES_START1;
			j = 0;
			break;

		case RES_START2:
			if(payloadIn[i] == RFE_START_BYTE_2)
				licenseProtocolState = RES_START3;
			else
				licenseProtocolState = RES_START1;
			break;

		case RES_START3:
			if(payloadIn[i] == RFE_START_BYTE_3)
				licenseProtocolState = RES_PAYLOAD;
			else
				licenseProtocolState = RES_START1;
			break;

		case RES_PAYLOAD:
			if(j < LICENSE_PAYLOAD_SIZE)
			{
				g_recvPayload[j] = payloadIn[i];
				j++;
			}
			else
			{
				uchar cs = 0;

				cs ^= RFE_START_BYTE_1 ^ RFE_START_BYTE_2 ^ RFE_START_BYTE_3;

				// compute Checksum
				for(j = 0;  j < LICENSE_PAYLOAD_SIZE; j++)
				{
					cs =  cs ^ g_recvPayload[j];
				}

				// if cs matches
				if(cs == payloadIn[i])
				{
#if ENCRYPTION_ON

#if DBG_LICENSE > 2
					TRACE_INFO("\n\r");
					TRACE_INFO("License:\t <R Crypt [%5d]: ", (uint)g_packageCounter);
					TRACE_INFO_ARRAY(g_recvPayload, 16);
#endif

					aes256_decrypt_ecb(&g_aesContext, g_recvPayload);
#endif
					licenseProtocolState = RES_START1;
					return true;
				}
				// no match --> error TODO:
				else
				{
					TRACE_ERROR("Checksum of the received package does not match! %02X != %02X\n\r", cs, payloadIn[i]);
					licenseProtocolState = RES_START1;
					return false;
				}
			}
			break;
		} // end switch
	} // end for

	licenseProtocolState = RES_START1;
	return false;
}


/*!
 * @brief computes the readerType from the buffer
 *
 * @param buff unsigned char pointer to the buffer structure where the incoming type is stored
 *
 * @return the readertype of the bootloader
 */
static ulong getReaderType ( uchar* buff )
{
	return ( (buff[0] << 24) | (buff[1] << 16) | (buff[2] << 8) | (buff[3] << 0) );
}



//------------------------------------------------------------------------------
//  	Global Functions
//------------------------------------------------------------------------------


void licProtocol_init ( )
{
}

bool licProtocol_atReset ( )
{
	return (g_state == CONNECTING) && (g_stateStep == 0);
}

void licProtocol_resetState()
{
	g_state = CONNECTING;
	g_stateStep = 0;
}

void licProtocol_startLiceneseProcess ( )
{
	TRACE_INFO("License:\t Sending StartBytes\n\r");

	g_packageCounter = 0;

	// init random
	rand_initialize();

	aes256_init(&g_aesContext, g_aesInitialKey); // init new random key from bootloader

	// special values; bootloader knows, that startpackage arrived
	fillRequestWithgetReaderTyp();

	SEND_PACKAGE();

	g_state = CONNECTING;
	g_stateStep = 0;
	g_used_readerId = 0;
	g_BL_readerType = 0;
}

void licProtocol_handleLicensePackage ( uchar* payloadIn, uint payloadIn_length )
{
	uint  i = 0;
	// contains the randVal needed by crypt and validation procedures
	static uchar randVal = 0;

	// TODO:
	//uchar *reportmessage = (uchar*) P_FLASH_REPORT;

	// Reset time out variable
	g_timeoutCounter = 0;

	if(parseLicenseMessage(payloadIn, payloadIn_length) == false)
	{
		TRACE_ERROR("License:\t parseLicenseMessage() ERROR!\n\r");
		return;
	}

#if DBG_LICENSE > 2
	TRACE_INFO("License:\t <R Plain [%5d]: ", (uint)g_packageCounter);
	TRACE_INFO_ARRAY(g_recvPayload, 16);
#endif

	switch(g_state)
	{
	case CONNECTING:
#if DBG_LICENSE > 1
		TRACE_INFO("License:\t CONNECTING\n\r");
#endif

		// send start package
		if( g_stateStep == 0 )
		{
			g_stateStep++;

			// get Bootloader Reader Type
			g_BL_readerType = getReaderType( &g_recvPayload[READER_TYPE_INDEX_0] );

			// no ID's left
			if( !licmgr_isLicenseAvailable( ) )
			{
				TRACE_ERROR("License:\t No ID's left!\n\r");
				RESET_STATE_MACHINE();
				return;
			}

			// check idStore Flag
			if( !licmgr_setLicenseMode(g_recvPayload[5]) )
			{
				TRACE_ERROR("License:\t Unknown License Mode!\n\r");
				RESET_STATE_MACHINE();
				return;
			}

			g_exchangeIDPackageCount = 0;

#if DBG_LICENSE
			TRACE_INFO("License:\t Received ReaderTyp %08X, Sending RandVal.\n\r", (uint)g_BL_readerType);
#endif

			fillRequestWithMagicRandVal( CONNECTING_VALIDATION_BYTE_1 );

			SEND_PACKAGE();
		}
		else if( g_stateStep == 1)
		{
			if(checkResponseWithMagicRandVal(RANDOM_INDEX_2, CONNECTING_VALIDATION_BYTE_2) == false)
				break;

#if DBG_LICENSE > 1
			TRACE_INFO("License:\t Validation OK -> Random Start\n\r");
#endif
			SWITCH_TO_RANDOM_START();
		}
		else
		{
			TRACE_FATAL("Unknown stateStep %d in State CONNECTING\n\r", g_stateStep);
			RESET_STATE_MACHINE();
		}

		break;

	case SWITCH_TO_BOOTLOADER_KEY:
#if DBG_LICENSE > 1
		TRACE_INFO("License:\t SWITCH_TO_BOOTLOADER_KEY\n\r");
#endif

		if( g_stateStep == 0 )
		{
			g_stateStep = 1;
			// overwrite packet with random values
			fillRequestWithRandom();

			SEND_PACKAGE();
		}
		// first part of changed AES key
		else if( g_stateStep == 1 )
		{
			g_stateStep = 2;
			// load first 16 Byte of new AES key from bootloader
			for(i = 0; i < AES_KEY_LENGTH_HALF; i++)
			{
				currentKey[i] = g_recvPayload[i];
			}

#if DBG_LICENSE
			TRACE_INFO("License:\t  AES_KEY (1): ");
			TRACE_INFO_ARRAY(currentKey, 16);
#endif

			fillRequestWithRandom();

			SEND_PACKAGE();
		}
		else if( g_stateStep == 2 )
		{
			g_stateStep = 3;

			for(i = 0; i < AES_KEY_LENGTH_HALF; i++)				// load second 16 Byte of new AES key from bootloader
			{
				currentKey[i + AES_KEY_LENGTH_HALF] = g_recvPayload[i];
			}

#if DBG_LICENSE
			TRACE_INFO("License:\t  AES_KEY (2): ");
			TRACE_INFO_ARRAY((&(currentKey[16])), 16);
#endif

			aes256_init(&g_aesContext, currentKey); // init new random key from bootloader

			fillRequestWithMagicRandVal(SWITCH_KEY_VALIDATION_BYTE_1);

			SEND_PACKAGE();
		}
		else if( g_stateStep == 3 )
		{
			if(checkResponseWithMagicRandVal(RANDOM_INDEX_2, SWITCH_KEY_VALIDATION_BYTE_2) == false)
				break;

#if DBG_LICENSE > 1
			TRACE_INFO("License:\t  Validation OK -> Random Start\n\r");
#endif
			SWITCH_TO_RANDOM_START();
		}
		else
		{
			TRACE_FATAL("Unknown stateStep %d in State SWITCH_TO_BOOTLOADER_KEY\n\r", g_stateStep);
			RESET_STATE_MACHINE();
		}
		break;

	case EXCHANGE_BLOCK_NR:
#if DBG_LICENSE > 1
		TRACE_INFO("License:\t EXCHANGE_BLOCK_NR\n\r");
#endif

		if( g_stateStep == 0  )
		{
			g_stateStep = 1;

			fillRequestWithMagicRandVal( EXCHANGE_BLK_NR_VALIDATION_BYTE_1 );

			SEND_PACKAGE();
		}
		else if( g_stateStep == 1 )
		{
			if(checkResponseWithMagicRandVal(RANDOM_INDEX_2 - 1, EXCHANGE_BLK_NR_VALIDATION_BYTE_2) == false)
				break;

#if DBG_LICENSE > 1
			TRACE_INFO("License:\t  Validation OK\n\r");
#endif

			// validation successfull and blockNr contains the block number
			g_exchangeBlockNr = g_recvPayload[BLOCK_NUMBER_XOR_CRYPT_INDEX] ^ BLOCK_NUMBER_XOR_CRYPT_VALUE;

			if( g_exchangeBlockNr >= BLOCK_NUMBER_MAX )
			{
				TRACE_ERROR("License:\t  Exchanged Block Number %2d is not valid!\n\r", g_exchangeBlockNr);
				RESET_STATE_MACHINE();
			}

#if DBG_LICENSE
			TRACE_INFO("License:\t  Exchanged Block Number: %2d and %2d -> Random Start\n\r", g_exchangeBlockNr, (g_exchangeBlockNr + BLOCK_NUMBER_OFFSET) % BLOCK_NUMBER_MAX);
#endif
			SWITCH_TO_RANDOM_START();
		}
		else
		{
			TRACE_FATAL("Unknown stateStep %d in State EXCHANGE_BLOCK_NR\n\r", g_stateStep);
			RESET_STATE_MACHINE();
		}
		break;

	case SWITCH_TO_LOCALRANDOM_KEY:
#if DBG_LICENSE > 1
		TRACE_INFO("License:\t SWITCH_TO_LOCALRANDOM_KEY\n\r");
#endif

		if( g_stateStep == 0 )
		{
			g_stateStep = 1;

			// create new random AES key
			// load new Key, but do not initialize AES -> current AES key is from bootloader
			fillAESdynamicKeyWithRandom(currentKey);

			// send first 16 Byte of new AES key
			for(i = 0; i < AES_KEY_LENGTH_HALF; i++)
			{
				g_sendBuff[i + 3] = currentKey[i];
			}

#if DBG_LICENSE
			TRACE_INFO("License:\t  AES_KEY (1): ");
			TRACE_INFO_ARRAY(g_sendBuff, 16);
#endif

			SEND_PACKAGE();
		}
		else if( g_stateStep == 1)
		{
			g_stateStep = 2;

			// send second 16 Byte of new AES key
			for (i = 0; i < AES_KEY_LENGTH_HALF; i++) 			// send second 16 Byte of new AES key
			{
				g_sendBuff[i + 3] = currentKey[i + AES_KEY_LENGTH_HALF];
			}

#if DBG_LICENSE
			TRACE_INFO("License:\t  AES_KEY (2): ");
			TRACE_INFO_ARRAY(g_sendBuff, 16);
#endif

			SEND_PACKAGE();
		}
		else if( g_stateStep == 2)
		{
			g_stateStep = 3;

			// test and certify new AES Key
			// init AES with a new random key from reader Id generator
			aes256_init(&g_aesContext, currentKey);

			fillRequestWithMagicRandVal(SWITCH_TO_LOCAL_VALIDATION_BYTE_1);

			SEND_PACKAGE();
		}
		else if(g_stateStep == 3)
		{
			if(checkResponseWithMagicRandVal(RANDOM_INDEX_2, SWITCH_TO_LOCAL_VALIDATION_BYTE_2) == false)
				break;

#if DBG_LICENSE > 1
			TRACE_INFO("License:\t  Validation OK -> Random Start\n\r");
#endif
			SWITCH_TO_RANDOM_START();
		}
		else
		{
			TRACE_FATAL("Unknown stateStep %d in State SWITCH_TO_LOCALRANDOM_KEY\n\r", g_stateStep);
			RESET_STATE_MACHINE();
		}
		break;

	case EXCHANGE_ADDITIONAL_INFORMATION:
#if DBG_LICENSE > 1
		TRACE_INFO("License:\t EXCHANGE_ADDITIONAL_INFORMATION\n\r");
#endif

		if( g_stateStep == 0 )
		{
			g_stateStep = 1;
			fillRequestWithRandom();
			SEND_PACKAGE();
		}
		else if(g_stateStep == 1)
		{
			if(licmgr_licenseMode() == READERID_AVAILABLE)
			{
				((uchar*)&g_used_readerId)[3] = g_recvPayload[1];
				((uchar*)&g_used_readerId)[2] = g_recvPayload[2];
				((uchar*)&g_used_readerId)[1] = g_recvPayload[3];
				((uchar*)&g_used_readerId)[0] = g_recvPayload[4];

#if DBG_LICENSE
				TRACE_INFO("License:\t Received the existing readerId: %08X\n\r", (uint)g_used_readerId);
#endif
			}
			SWITCH_TO_RANDOM_START();
		}
		else
		{
			TRACE_FATAL("Unknown stateStep %d in State EXCHANGE_ADDITIONAL_INFORMATION\n\r", g_stateStep);
			RESET_STATE_MACHINE();
		}

		break;

	case EXCHANGE_ID:
#if DBG_LICENSE > 1
		TRACE_INFO("License:\t EXCHANGE_ID\n\r");
#endif

		// In this state, we exchange the reader ID. For this we send 50 verifying packets mutually.
		// In two packets (block number and (block number + BLOCK_NUMBER_OFFSET modulo BLOCK_NUMBER_MAX) there is new reader ID.
		// In packet 45, I write in the flash the readerIDcount++ and readerID++.
		// If Bootloader not answering (packet 45). I made flash modifications back (readerIDcount-- and readerID--).
		// From answer Packet 47 until to the end, we were successful. But I send 50 Packets to the bootloader

		if( g_stateStep == 0 )
		{
			g_stateStep = 1;

			fillRequestWithMagicRandVal(EXCHANGE_ID_VALIDATION_BYTE_1);

			SEND_PACKAGE();
		}
		else if( g_stateStep == 1 )
		{
			++g_exchangeIDPackageCount;
#if DBG_LICENSE
			TRACE_INFO("License:\t  Packet Count - %2d\n\r", g_exchangeIDPackageCount);
#endif

			if(checkResponseWithMagicRandVal(RANDOM_INDEX_2 , EXCHANGE_ID_VALIDATION_BYTE_2) == false)
				break;

			fillRequestWithRandom();
			randVal = rand_getSpecificRandom(1,7);

			// write flags and reduce id count
			if(g_exchangeIDPackageCount == LICENSE_EX_ID_PACKAGE_COUNT - 5)	// write succesfull in the flash and afterwards verify with bootloader
			{
#if DBG_LICENSE
				TRACE_INFO("License:\t  Storing Data to flash\n\r");
#endif

				ulong flags = 0;
				flags |= (licmgr_licenseMode() == READERID_AVAILABLE)?0:FLAG_NEW_ID_ASSIGNED;

				report_writeSuccessReport(g_BL_readerType, g_used_readerId, flags);

				licmgr_storeData();
			}
			else if(g_exchangeIDPackageCount == LICENSE_EX_ID_PACKAGE_COUNT)
			{
#if DBG_LICENSE
				TRACE_INFO("License:\t  Finished - Resetting State Machine\n\r");
#endif
				protocol_sendLicenseResult(true);
				RESET_STATE_MACHINE();
			}
			// send reader id if blocknumber fits
			else if(g_exchangeIDPackageCount == g_exchangeBlockNr || (g_exchangeIDPackageCount == ((g_exchangeBlockNr + BLOCK_NUMBER_OFFSET) % BLOCK_NUMBER_MAX)))
			{
				ulong readerId = 0;
				// get new license if available
				if( !licmgr_getNextReaderId( &readerId ) )
				{
					TRACE_ERROR("License:\t getNextLicense failed!\n\r");
					RESET_STATE_MACHINE();
					break;
				}

#if DBG_LICENSE
				TRACE_INFO("License:\t  Exchanging ID: 0x%08X \n\r", (uint)readerId);
#endif
				randVal = 7;

				// crypt and store in buffer
				fillRequestWithEncryptedReaderID(&readerId );

				if(licmgr_licenseMode() == READERID_NEEDED)
					g_used_readerId = readerId;
			}

			// verify answer every time
			g_sendPayload[RANDOM_INDEX_1] = randVal;
			g_sendPayload[RANDOM_INDEX_1 + randVal] = (uchar) (( EXCHANGE_ID_VALIDATION_BYTE_1 ) - (randVal << RANDVAL_SHIFT_VALUE));

			SEND_PACKAGE();

		} // end if( packageState == 0)
		else
		{
			TRACE_FATAL("Unknown stateStep %d in State EXCHANGE_ID\n\r", g_stateStep);
			RESET_STATE_MACHINE();
		}

		break;

	case RANDOM_START:
#if DBG_LICENSE > 1
		TRACE_INFO("License:\t RANDOM_START\n\r");
#endif

		if ( g_recvPayload[RANDOM_INDEX_4] - RANDOM_OFFSET != g_randomRunCounter )
		{
			TRACE_ERROR("License:\t  Wrong Validation values! %02X != %02X\n\r", g_recvPayload[RANDOM_INDEX_4]	- RANDOM_OFFSET, g_randomRunCounter);
			//setReport(RANDOM_START,WRONGVERIFY, 0xFF, 0xFF);
			RESET_STATE_MACHINE();
			break;
		}
		g_state = RANDOM;

	case RANDOM:
#if DBG_LICENSE > 1
		TRACE_INFO("License:\t RANDOM\n\r");
#endif

		if ( g_randomRunCounter == 1 )	// go in next state, if random run counter reached 0
		{
			switch ( g_stateBeforeRandom )
			{
			case CONNECTING:
				g_state = SWITCH_TO_BOOTLOADER_KEY;
				break;

			case SWITCH_TO_BOOTLOADER_KEY:
				g_state = EXCHANGE_BLOCK_NR;
				break;

			case EXCHANGE_BLOCK_NR:
				g_state = SWITCH_TO_LOCALRANDOM_KEY;
				break;

			case SWITCH_TO_LOCALRANDOM_KEY:
				g_state = EXCHANGE_ADDITIONAL_INFORMATION;
				break;

			case EXCHANGE_ADDITIONAL_INFORMATION:
				g_state = EXCHANGE_ID;
				break;

			case EXCHANGE_ID:
				g_state = CONNECTING;
				break;

			default:
				g_state = CONNECTING;
				break;
			}
			g_stateStep = 0;
		}

		// send only random packages in this state
		fillRequestWithRandom();
		SEND_PACKAGE();
		// one random packet is forwarded
		g_randomRunCounter--;

		break;

	} // end switch

}
