/*!
 * @file	readerHostProtocol.c
 * @brief	Platform independent implementation of the reader-host-protocol
 */

#include "readerHostProtocol.h"

#include <utility/trace.h>

#include "readerHostProtocolComm.h"
#include "readerHostProtocolDefines.h"

#include <core/ringBuffer/buffer.h>

/*! @brief Type definition for function pointer to a request handler */
typedef 	eRFE_RET_VALUE(*cfp)(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);

/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_getSerialNumber(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);
/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_getReaderType(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);
/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_getHardwareRevision(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);
/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_getSoftwareRevision(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);


/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_passThrough(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);
/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_startLicenseProcedure(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);
/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_getLicenseInfo(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);
/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_getReportInfo(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);
/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_getReportData(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);
/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern eRFE_RET_VALUE 	protocolImpl_getErrorReport(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length);

/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
extern void 			protocolImpl_process ( void );


/*! @brief Standard request handler for unkown command */
static eRFE_RET_VALUE unkownCommand(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length)
{
	return RFE_RET_ERR_OP_NOT_SUPPORTED;
}

/*!
 * @brief Command parser
 *
 * This function parses the two sent commands and returns the appropriate request handler.
 * @param 	command1	The first command
 * @param 	command2 	The second command
 * @return	Returns the appropriate request handler.
 */
static cfp commandParser(uchar command1, uchar command2)
{
	switch(command1)
	{
		case RFE_COM1_READER_COMMON:
			switch(command2)
			{
				case RFE_COM2_GET_SERIAL_NUMBER:
					return protocolImpl_getSerialNumber;
				case RFE_COM2_GET_READER_TYPE:
					return protocolImpl_getReaderType;
				case RFE_COM2_GET_HARDWARE_REVISION:
					return protocolImpl_getHardwareRevision;
				case RFE_COM2_GET_SOFTWARE_REVISION:
					return protocolImpl_getSoftwareRevision;
				default:
					break;
			}
		case RFE_COM1_LICENSE:

			switch(command2)
			{
				case RFE_COM2_PASS_THROUGH:
					return protocolImpl_passThrough;
				case RFE_COM2_START_LICENSE:
					return protocolImpl_startLicenseProcedure;
				case RFE_COM2_GET_LICENSE_INFO:
					return protocolImpl_getLicenseInfo;
				case RFE_COM2_GET_REPORT_INFO:
					return protocolImpl_getReportInfo;
				case RFE_COM2_GET_REPORT_DATA:
					return protocolImpl_getReportData;
				case RFE_COM2_GET_ERROR_REPORT:
					return protocolImpl_getErrorReport;
				default:
					break;
			}

		default:
			return unkownCommand;
	}
	return unkownCommand;
}

/*!
 * @brief Calculates the checksum
 *
 * This function calculates the checksum for the given data. The checksum is just a simple XOR conjuction.
 * @param	data		Pointer to the data
 * @param	length		Length of the data
 * @return	Returns the calculated checksum.
 */
static uchar calcChecksum (uchar* data, uchar length)
{
	uchar cs = 0;
	int i;
	for(i = 0; i < length; i++)
		cs ^= (uchar)data[i];
	return cs;
}

/*!
 * @brief Sends a protocol conform package to host
 *
 * The function sends a prtocol conform package to host. It adds the start bytes, and the checksum.
 * @param 	payload			Pointer to the payload that should be transmitted
 * @param	payloadLength	Length of the payload
 * @param	command1		The first command
 * @param	command2		The second command
 * @return	Returns true if the operation was successful.
 */
bool sendPackageToHost(uchar* payload, uint payloadLength, uchar command1, uchar command2)
{
	static uchar sendBuffer[SENDBUFFER_SIZE];
	uint i = 0;

	// Calculate the payload length
	uchar packetLength = payloadLength + RFE_DATA_BASE_INDEX + 2;

	// Reset the sendbuffer
	for(i = 0; i < SENDBUFFER_SIZE; i++)
		sendBuffer[i] = 0;

	// Build up the frame for the packet
	sendBuffer[RFE_START_INDEX_1] = RFE_START_BYTE_1;
	sendBuffer[RFE_START_INDEX_2] = RFE_START_BYTE_2;
	sendBuffer[RFE_START_INDEX_3] = RFE_START_BYTE_3;
	sendBuffer[RFE_COMMAND_START_INDEX] = RFE_COMMAND_START_BYTE;
	sendBuffer[RFE_COMMAND_INDEX_1] = command1;
	sendBuffer[RFE_COMMAND_INDEX_2] = command2;
	sendBuffer[RFE_LENGTH_START_INDEX] = RFE_LENGTH_START_BYTE;
	sendBuffer[RFE_LENGTH_INDEX] = payloadLength;
	sendBuffer[RFE_PAYLOAD_START_INDEX] = RFE_PAYLOAD_START_BYTE;

	// Copy the paylod into the frame
	for(i = 0; i < payloadLength; i++)
		sendBuffer[RFE_PAYLOAD_INDEX + i] = payload[i];

	// Add the checksum
	sendBuffer[packetLength-2] = RFE_CHECKSUM_START_BYTE;
	sendBuffer[packetLength-1] = calcChecksum(sendBuffer, packetLength-1);

	// Send the packet
	return (packetLength == protocolComm_sendBytes(sendBuffer, packetLength));
}


/*!
 * @brief Message Parser
 *
 * This function checks if data is available to be read. If data is available, the data is parsed. If the data is protocol
 * conform the request handler for this request is called. If the request handler wants to send an answer the answer is sent.
 */
static void messageParser()
{
	static uchar command1 = 0, command2 = 0;
	static eMessageState state = START_BYTE_1;

	static uint 	payloadIn_length = 0;
	static uint 	payloadIn_index = 0;
	static uchar 	payloadIn[RHP_PAYLOAD_SIZE];

	static uint  	payloadOut_length = 0;
	static uchar 	payloadOut[RHP_PAYLOAD_SIZE];


	cfp handler;
	uint i;

	// Get the count of bytes to read
	uint bytes2Read = protocolComm_bytesAvailable();

	for(i =0 ; i < bytes2Read; i++)
	{
		uchar c;

		// Get the next byte
		if(buffer_readByte(&c) == false)
			return;

#if DBG_RHPROT > 1
		TRACE_DEBUG("RHProt:\t Read byte: 0x%02x\n\r", c);
#endif

		switch(state){
			case START_BYTE_1:
				if(c == RFE_START_BYTE_1)
				{
					state = START_BYTE_2;
#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t -> Got Start Byte 1\n\r");
#endif
				}
				break;

			case START_BYTE_2:
				if(c == RFE_START_BYTE_2)
				{
					state = START_BYTE_3;
#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t -> Got Start Byte 2\n\r");
#endif
				}
				else
				{
					state = START_BYTE_1;
				}
				break;

			case START_BYTE_3:
				if(c == RFE_START_BYTE_3)
				{
					state = COMMAND_START_BYTE;
#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t -> Got Start Byte 3\n\r");
#endif
				}
				else
				{
					state = START_BYTE_1;
				}
				break;

			case COMMAND_START_BYTE:
				if(c == RFE_COMMAND_START_BYTE)
				{
					state = COMMAND_1;
#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t -> Command Start Byte\n\r");
#endif
				}
				else
				{
					state = START_BYTE_1;
				}
				break;

			case COMMAND_1:
				command1 = c;
				state = COMMAND_2;
#if DBG_RHPROT
				TRACE_DEBUG("RHProt:\t -> Command1 is %02X\n\r", c);
#endif
				break;

			case COMMAND_2:
				command2 = c;
				state = LENGTH_START_BYTE;
#if DBG_RHPROT
				TRACE_DEBUG("RHProt:\t -> Command2 is %02X\n\r", c);
#endif
				break;

			case LENGTH_START_BYTE:
				if(c == RFE_LENGTH_START_BYTE)
				{
					state = LENGTH;
#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t -> Length Start Byte\n\r");
#endif
				}
				else
				{
					state = START_BYTE_1;
				}
				break;

			case LENGTH:
				payloadIn_length = c;
				payloadIn_index = 0;
				if(payloadIn_length == 0)
				{
					state = CHECKSUM_START_BYTE;
#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t -> Length is %02X\n\r", c);
#endif
				}
				else
				{
					state = PAYLOAD_START_BYTE;
				}
				break;

			case PAYLOAD_START_BYTE:
				if(c == RFE_PAYLOAD_START_BYTE)
				{
					state = PAYLOAD;
#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t -> Payload Start Byte\n\r");
#endif
				}
				else
				{
					state = START_BYTE_1;
				}
				break;

			case PAYLOAD:
				payloadIn[payloadIn_index++] = c;
				if(payloadIn_index == payloadIn_length)
				{
					state = CHECKSUM_START_BYTE;
				}
#if DBG_RHPROT
				TRACE_DEBUG("RHProt:\t -> Payload Byte %02X\n\r", c);
#endif
				break;

			case CHECKSUM_START_BYTE:
				if(c == RFE_CHECKSUM_START_BYTE)
				{
					state = CHECKSUM;
#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t -> Checksum Start Byte\n\r");
#endif
				}
				else
				{
					state = START_BYTE_1;
				}
				break;

			case CHECKSUM:
				{
					eRFE_RET_VALUE result;

					// Calculate the checksum of the payload
					uchar cs = calcChecksum(payloadIn, payloadIn_length);

					// Add the checksum of the rest of the frame
					cs ^= RFE_START_BYTE_1 ^ RFE_START_BYTE_2 ^ RFE_START_BYTE_3 ^
							RFE_COMMAND_START_BYTE ^ RFE_LENGTH_START_BYTE ^ RFE_CHECKSUM_START_BYTE; 		// xor sum of known start bytes
					cs ^= payloadIn_length;
					cs ^= command1;
					cs ^= command2;
					if(payloadIn_length != 0)	// if payload follows xor payload_start_byte
						cs ^= RFE_PAYLOAD_START_BYTE;

					// Reset the state
					state = START_BYTE_1;

					// Check if the checksum matches
					if(c != cs)
					{
#if DBG_RHPROT
						TRACE_DEBUG("RHProt:\t Got wrong checksum. Calculated %02X - Got %02X\n\r", cs, c);
#endif
						break;
					}

#if DBG_RHPROT
					TRACE_DEBUG("RHProt:\t Checksum OK\n\r");
#endif
					// Get the appropriate request handler
					handler = commandParser(command1, command2);

					// Call the request handler
					payloadOut_length = 0;
					result = handler(payloadIn, payloadIn_length, payloadOut, &payloadOut_length);

					// If there is nothing to do, stop
					if(result == RFE_RET_NOTHING_TODO)
						break;

					// If the return code is an error just send the return code
					else if(result != RFE_RET_SUCCESS)
					{
						payloadOut_length = 1;
						payloadOut[0] = result;
					}

					sendPackageToHost(payloadOut, payloadOut_length, command1, command2);

					break;
				} // end case CHECKSUM
		} // end switch
	} // end for
} // end processMessage()


/*!
 * @brief Function to check if anything to do
 *
 * This function checks if bytes are available to be read from the buffer. If bytes are available the message parser is called.
 * @return 	Ignore, always returns true
 */
static bool receiveFromHost()
{
	if (protocolComm_bytesAvailable() > 0)
	{
		messageParser();
	}
	return true;
}


/*!
 * @brief Central process function
 *
 * This function is the central process function. It calls the process function of the communication interface. After this
 * it is checked if something can be read. After this the process function of the implementation is called.
 */
void protocol_process()
{
	// Call the process function of the communication
	protocolComm_processCommunication();

	// See if there is something to read
	receiveFromHost();

	// Call the process function of the implementation
	protocolImpl_process();

}




