/*!
 * @file 	readerHostProtocolComm.c
 * @brief	The platform independent implementation of the communication module
 */

#include "readerHostProtocolComm.h"

#include <utility/trace.h>

#include <hal/comm/cdc/commCDC.h>

#include <core/ringBuffer/buffer.h>

/*! @brief Type definition for a function pointer to a write function every interface has to implement */
typedef uint (*fpWriteFunction)		(uchar* buffer, uint size);
/*! @brief Type definition for a function pointer to a process function every interface has to implement */
typedef void (*fpProcessFunction)	(void);

/*! @brief Defines the maximal interface count that can be configured */
#define MAX_INTERFACE_COUNT 	4


/*! @brief Counter of the configured interfaces */
static uchar interfaceCount = 0;

/*! @brief Array with function pointer to the write functions of every configured interface */
static fpWriteFunction 		readerHostCommWriteFunction[MAX_INTERFACE_COUNT];
/*! @brief Array with function pointer to the process functions of every configured interface */
static fpProcessFunction	readerHostCommProcessFunction[MAX_INTERFACE_COUNT];


/*!
 * @brief Initialize function for all interfaces
 *
 * This function initializes every given interface. It stores the pointer to their write and process functions.
 * @param	interface		A bit mask with the interfaces that should be used
 * @see 	READER_HOST_SERIAL_COMM_INTERFACE
 * @see 	READER_HOST_CDC_COMM_INTERFACE
 * @see 	READER_HOST_HID_COMM_INTERFACE
 * @see 	READER_HOST_ETHERNET_COM_INTERFACE
 */
void protocolComm_configureCommunication ( void )
{
#if DBG_RHPROT
	TRACE_DEBUG("RHCom:\t Configuring ReaderHostComm.\n\r");
#endif

	// Initialize the cdc interface
#if DBG_RHPROT
		TRACE_DEBUG("RHCom:\t Configuring ReaderHostComm with CDC communication.\n\r");
#endif
		cdc_init();
		readerHostCommWriteFunction[interfaceCount] 	= &cdc_write;
		readerHostCommProcessFunction[interfaceCount] 	= &cdc_process;
		interfaceCount++;

}

/*!
 * @brief Process function of Communication
 *
 * The process function of the communication implementation calls the process functions of every configured interface.
 */
void protocolComm_processCommunication ( void )
{
	uchar i = 0;

	// Call the process function of every configured interface
	for(i = 0; i < interfaceCount; i++){
		readerHostCommProcessFunction[i]();
	}
}

/*!
 * @brief Write function
 *
 * The write function of the communication implementation writes the given bytes to every configured interface.
 * @param	buffer		Pointer to the data that should be sent
 * @param 	size		Length of the data
 * @return	Returns the bytes that were written. Always returns size.
 */
uint protocolComm_sendBytes ( uchar* buffer, uint size )
{
	uchar i = 0;

	// Write the bytes to every configured interface
	for(i = 0; i < interfaceCount; i++){
		readerHostCommWriteFunction[i](buffer, size);
	}
	return size;
}

/*!
 * @brief Check function if bytes are available
 *
 * This function checks the ring buffer if there are bytes available that can be read.
 * @return 	Returns the count of bytes that are available to be read.
 */
uint protocolComm_bytesAvailable (void)
{
	return buffer_bytesToRead();
}

/*!
 * @brief Single read function
 *
 * Reads one byte from the buffer and returns it.
 * @return	Returns the read byte.
 */
uchar protocolComm_readByte()
{
	uchar c = 0xFF;
	buffer_readByte(&c);
	return c;
}

/*!
 * @brief Multiple read function
 *
 * Reads the avaiable bytes from the buffer into an other buffer.
 * @param	buffer		The destination buffer
 * @param 	size		The count of data that should be read
 * @return Returns the count of read bytes.
 */
uint protocolComm_readBytesIntoBuffer(uchar* buffer, uint size)
{
	return buffer_readBytes(buffer, size);
}
