/*!
 * @file 	readerHostProtocolImpl.c
 * @brief	The platform dependent implementation of the reader-host-protocol
 */

#include "readerHostProtocol.h"
#include "readerHostProtocolImpl.h"

#include "readerHostProtocolComm.h"
#include "readerHostProtocolDefines.h"

#include <core/crypt/aes256.h>
#include <utility/trace.h>

#include <hal/boardLed.h>

#include "licenseProtocol.h"
#include <core/licenseMgr/licenseMgr.h>
#include <core/report/report.h>
#include <core/store/store.h>

//------------------------------------------------------------------------------
//  	Global Variables & Functions
//------------------------------------------------------------------------------

/*! @brief Forward declaration of the reader-host-protocol */
extern bool sendPackageToHost(uchar* payload, uint payloadLength, uchar command1, uchar command2);

//------------------------------------------------------------------------------
//  	Local Functions
//------------------------------------------------------------------------------

/*!
 * @brief Initialization of the protocol
 *
 * The function to initialize the protocol. Here the reader-host-communication is initialized with the interfaces
 * specified in the Makefile. After this the READ_LED is initialized.
 */
bool protocol_init ( void )
{
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Initializing RHP.\n\r");
#endif

	// Init the read LED
	INIT_INFO_LED();

	SET_INFO_LED();

	protocolComm_configureCommunication();

	CLEAR_INFO_LED();

	return true;
}


void protocol_sendLicenseResult ( bool result )
{
	static uchar buffer[1];

	bool ret;
	uchar len = 0;

	buffer[len++] = result;

	// Send the package
	ret = sendPackageToHost(buffer, len, RFE_COM1_LICENSE, RFE_COM2_LICENSE_RESULT_INTERRUPT);

	return;
}


/*!
 * @brief Reader-Host Get Serial Number
 *
 * Function that is called by the reader-host-protocol. It handles the request for the serial number.
 * @param 	payloadIn				Pointer to the payload that was sent from the host
 * @param 	payloadIn_length		Length of the payload that was sent from the host
 * @param 	payloadOut				Pointer to the buffer where the answer can be written in
 * @param 	payloadOut_length		Length of the answer payload
 * @return	Returns the result of the request. If RFE_RET_SUCCESS is returned the package should be built up and sent. If
 * RFE_RET_NOTHING_TODO is returned, nothing should be sent. If any other code is returned, only this code should be transmitted
 * to the host.
 */
eRFE_RET_VALUE protocolImpl_getSerialNumber ( uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length )
{
	uint serialNumber;
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Get Serial Number.\n\r");
#endif

	serialNumber = store_orderNumber();

	*payloadOut_length = sizeof(serialNumber);
	payloadOut[0] = ((uchar*)&serialNumber)[3];
	payloadOut[1] = ((uchar*)&serialNumber)[2];
	payloadOut[2] = ((uchar*)&serialNumber)[1];
	payloadOut[3] = ((uchar*)&serialNumber)[0];

	return RFE_RET_SUCCESS;
}

/*!
 * @brief Reader-Host Get Reader Type
 *
 * Function that is called by the reader-host-protocol. It handles the request for the reader type.
 * @param 	payloadIn				Pointer to the payload that was sent from the host
 * @param 	payloadIn_length		Length of the payload that was sent from the host
 * @param 	payloadOut				Pointer to the buffer where the answer can be written in
 * @param 	payloadOut_length		Length of the answer payload
 * @return	Returns the result of the request. If RFE_RET_SUCCESS is returned the package should be built up and sent. If
 * RFE_RET_NOTHING_TODO is returned, nothing should be sent. If any other code is returned, only this code should be transmitted
 * to the host.
 */
eRFE_RET_VALUE protocolImpl_getReaderType ( uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length )
{
	uint readerType;
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Get Reader Type.\n\r");
#endif

	readerType = READER_TYPE;

	*payloadOut_length = sizeof(readerType);
	payloadOut[0] = ((uchar*)&readerType)[3];
	payloadOut[1] = ((uchar*)&readerType)[2];
	payloadOut[2] = ((uchar*)&readerType)[1];
	payloadOut[3] = ((uchar*)&readerType)[0];

	return RFE_RET_SUCCESS;
}

/*!
 * @brief Reader-Host Get Hardware Revision
 *
 * Function that is called by the reader-host-protocol. It handles the request for the hardware revision.
 * @param 	payloadIn				Pointer to the payload that was sent from the host
 * @param 	payloadIn_length		Length of the payload that was sent from the host
 * @param 	payloadOut				Pointer to the buffer where the answer can be written in
 * @param 	payloadOut_length		Length of the answer payload
 * @return	Returns the result of the request. If RFE_RET_SUCCESS is returned the package should be built up and sent. If
 * RFE_RET_NOTHING_TODO is returned, nothing should be sent. If any other code is returned, only this code should be transmitted
 * to the host.
 */
eRFE_RET_VALUE protocolImpl_getHardwareRevision ( uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length )
{
	uint hwRevision;
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Get Hardware Number.\n\r");
#endif
	hwRevision = HARDWARE_VERSION;

	*payloadOut_length = sizeof(hwRevision);
	payloadOut[0] = ((uchar*)&hwRevision)[3];
	payloadOut[1] = ((uchar*)&hwRevision)[2];
	payloadOut[2] = ((uchar*)&hwRevision)[1];
	payloadOut[3] = ((uchar*)&hwRevision)[0];

	return RFE_RET_SUCCESS;
}

/*!
 * @brief Reader-Host Get Software Revision
 *
 * Function that is called by the reader-host-protocol. It handles the request for the software revision.
 * @param 	payloadIn				Pointer to the payload that was sent from the host
 * @param 	payloadIn_length		Length of the payload that was sent from the host
 * @param 	payloadOut				Pointer to the buffer where the answer can be written in
 * @param 	payloadOut_length		Length of the answer payload
 * @return	Returns the result of the request. If RFE_RET_SUCCESS is returned the package should be built up and sent. If
 * RFE_RET_NOTHING_TODO is returned, nothing should be sent. If any other code is returned, only this code should be transmitted
 * to the host.
 */
eRFE_RET_VALUE protocolImpl_getSoftwareRevision ( uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length )
{
	uint swRevision;
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Get Software Revision.\n\r");
#endif

	swRevision = SOFTWARE_VERSION;

	*payloadOut_length = sizeof(swRevision);
	payloadOut[0] = ((uchar*)&swRevision)[3];
	payloadOut[1] = ((uchar*)&swRevision)[2];
	payloadOut[2] = ((uchar*)&swRevision)[1];
	payloadOut[3] = ((uchar*)&swRevision)[0];

	return RFE_RET_SUCCESS;
}


/*!
 * @brief Reader-Host parse through
 *
 * Function that is called by the reader-host-protocol. It handles the request for the serial number.
 * @param 	payloadIn				Pointer to the payload that was sent from the host
 * @param 	payloadIn_length		Length of the payload that was sent from the host
 * @param 	payloadOut				Pointer to the buffer where the answer can be written in
 * @param 	payloadOut_length		Length of the answer payload
 * @return	Returns the result of the request. If RFE_RET_SUCCESS is returned the package should be built up and sent. If
 * RFE_RET_NOTHING_TODO is returned, nothing should be sent. If any other code is returned, only this code should be transmitted
 * to the host.
 */
eRFE_RET_VALUE protocolImpl_passThrough ( uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length )
{
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Pass Through\n\r");
#endif

	licProtocol_handleLicensePackage(payloadIn, payloadIn_length);

	return RFE_RET_NOTHING_TODO;
}

/*!
 * @brief Reader-Host start license procedure
 *
 * Function that is called by the reader-host-protocol. It handles the request for the serial number.
 * @param 	payloadIn				Pointer to the payload that was sent from the host
 * @param 	payloadIn_length		Length of the payload that was sent from the host
 * @param 	payloadOut				Pointer to the buffer where the answer can be written in
 * @param 	payloadOut_length		Length of the answer payload
 * @return	Returns the result of the request. If RFE_RET_SUCCESS is returned the package should be built up and sent. If
 * RFE_RET_NOTHING_TODO is returned, nothing should be sent. If any other code is returned, only this code should be transmitted
 * to the host.
 */
eRFE_RET_VALUE protocolImpl_startLicenseProcedure ( uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length )
{
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Start License Procedure\n\r");
#endif

	licProtocol_startLiceneseProcess();

	*payloadOut_length = 1;
	payloadOut[0] = RFE_RET_SUCCESS;

	return RFE_RET_SUCCESS;
}

/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
eRFE_RET_VALUE 	protocolImpl_getLicenseInfo(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length)
{
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Get License Info\n\r");
#endif
	uint index = 0;
	ulong startId, startCount, nextId, remainingCount;

	licmgr_getLicenseInfo(&startId, &startCount, &nextId, &remainingCount);

	*payloadOut_length = sizeof(startId) + sizeof(startCount) + sizeof(nextId) + sizeof(remainingCount);

	payloadOut[index++] = ((uchar*)&startId)[3];
	payloadOut[index++] = ((uchar*)&startId)[2];
	payloadOut[index++] = ((uchar*)&startId)[1];
	payloadOut[index++] = ((uchar*)&startId)[0];

	payloadOut[index++] = ((uchar*)&startCount)[3];
	payloadOut[index++] = ((uchar*)&startCount)[2];
	payloadOut[index++] = ((uchar*)&startCount)[1];
	payloadOut[index++] = ((uchar*)&startCount)[0];

	payloadOut[index++] = ((uchar*)&nextId)[3];
	payloadOut[index++] = ((uchar*)&nextId)[2];
	payloadOut[index++] = ((uchar*)&nextId)[1];
	payloadOut[index++] = ((uchar*)&nextId)[0];

	payloadOut[index++] = ((uchar*)&remainingCount)[3];
	payloadOut[index++] = ((uchar*)&remainingCount)[2];
	payloadOut[index++] = ((uchar*)&remainingCount)[1];
	payloadOut[index++] = ((uchar*)&remainingCount)[0];

	return RFE_RET_SUCCESS;
}

/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
eRFE_RET_VALUE 	protocolImpl_getReportInfo(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length)
{
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Get Report Info\n\r");
#endif
	uint index = 0;
	ulong reportCount = 0;
	reportCount = report_getReportCount();

	*payloadOut_length = sizeof(reportCount);

	payloadOut[index++] = ((uchar*)&reportCount)[3];
	payloadOut[index++] = ((uchar*)&reportCount)[2];
	payloadOut[index++] = ((uchar*)&reportCount)[1];
	payloadOut[index++] = ((uchar*)&reportCount)[0];

	return RFE_RET_SUCCESS;
}

/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
eRFE_RET_VALUE 	protocolImpl_getReportData(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length)
{
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Get Report Data\n\r");
#endif
	uint index = 0;
	ulong reportIndex = 0;

	if(payloadIn_length != 4)
		return RFE_RET_ERR_WRONG_PARAM_COUNT;

	reportIndex += (((ulong)payloadIn[index++]) << 24);
	reportIndex += (((ulong)payloadIn[index++]) << 16);
	reportIndex += (((ulong)payloadIn[index++]) << 8);
	reportIndex += (((ulong)payloadIn[index++]));

	index = 0;
	ulong readerId = 0, readerType = 0, flags = 0;
	report_getReport(reportIndex, &readerId, &readerType, &flags);

	*payloadOut_length = sizeof(readerId) + sizeof(readerType) + sizeof(flags);

	payloadOut[index++] = ((uchar*)&readerId)[3];
	payloadOut[index++] = ((uchar*)&readerId)[2];
	payloadOut[index++] = ((uchar*)&readerId)[1];
	payloadOut[index++] = ((uchar*)&readerId)[0];

	payloadOut[index++] = ((uchar*)&readerType)[3];
	payloadOut[index++] = ((uchar*)&readerType)[2];
	payloadOut[index++] = ((uchar*)&readerType)[1];
	payloadOut[index++] = ((uchar*)&readerType)[0];

	payloadOut[index++] = ((uchar*)&flags)[3];
	payloadOut[index++] = ((uchar*)&flags)[2];
	payloadOut[index++] = ((uchar*)&flags)[1];
	payloadOut[index++] = ((uchar*)&flags)[0];

	return RFE_RET_SUCCESS;
}

/*! @brief Forward declaration of request handler, implemented in the platform dependent implementation */
eRFE_RET_VALUE 	protocolImpl_getErrorReport(uchar* payloadIn, uint payloadIn_length, uchar* payloadOut, uint* payloadOut_length)
{
#if DBG_RHIMPL
	TRACE_DEBUG("RHImpl:\t Host -> Get Error Report\n\r");
#endif

	return RFE_RET_ERR_OP_NOT_SUPPORTED;
}



/*!
 * @brief Process function of reader-host-implementation
 *
 * The process function of the reader-host-implementation. This function is called continuously. It controls the call
 * of the inventory and heartbeat functions. It also checks the as399x if it is ok. Also the frequency hopping is controlled
 * from here.
 */
void protocolImpl_process ( void )
{
	//if(g_resetDevice == true){}
}
