/*
 * report.c
 *
 *  Created on: 16.03.2010
 *      Author: stefan.detter
 */

#include <utility/trace.h>
#include <core/store/store.h>
#include <def.h>

#include "report.h"

//------------------------------------------------------------------------------
//  	Variables
//------------------------------------------------------------------------------
#define REPORT_INFO_FLAG_OK			0x55AA11EE

typedef struct
{
	ulong reportCount;
	ulong nextReportAddress;
} REPORT_INFO;

static REPORT_INFO		g_reportInfo;


typedef struct
{
	ulong magicRandValErrorCount;
	ulong timeOutErrorCount;
} ERROR_REPORT;

static ERROR_REPORT		g_errorReport;


typedef struct
{
	ulong readerId;
	ulong readerType;
	ulong flags;
} SUCCESS_REPORT;



#define SAVE_REPORT_INFO()			store_writeData(REPORT_INFO_ADDRESS + 4, (uchar*)&g_reportInfo, sizeof(REPORT_INFO))
#define LOAD_REPORT_INFO()			store_readData(REPORT_INFO_ADDRESS + 4, (uchar*)&g_reportInfo, sizeof(REPORT_INFO))

#define SAVE_ERROR_REPORT()			store_writeData(REPORT_INFO_ADDRESS + 4 + sizeof(REPORT_INFO), (uchar*)&g_errorReport, sizeof(ERROR_REPORT))
#define LOAD_ERROR_REPORT()			store_readData(REPORT_INFO_ADDRESS + 4 + sizeof(REPORT_INFO), (uchar*)&g_errorReport, sizeof(ERROR_REPORT))

static void dumpReports()
{
	LOAD_REPORT_INFO();
	LOAD_ERROR_REPORT();

#if DBG_REPORT
	TRACE_INFO("Report:\t Current ReportInfo:\n\r");
	TRACE_INFO("Report:\t \tReport Count:           %d\n\r", (uint)g_reportInfo.reportCount);
	TRACE_INFO("Report:\t \tNext Report Address:    %08X\n\r", (uint)g_reportInfo.nextReportAddress);

	TRACE_INFO("Report:\t Current ErrorReport:\n\r");
	TRACE_INFO("Report:\t \tMagic RandVal Error:    %d\n\r", (uint)g_errorReport.magicRandValErrorCount);
	TRACE_INFO("Report:\t \tTime Out Error:         %d\n\r", (uint)g_errorReport.timeOutErrorCount);
#endif

#if DBG_REPORT > 1
	TRACE_INFO("Report:\t Current SuccessReports:\n\r");
	ulong i = 0;
	ulong addrOffset = 0;
	SUCCESS_REPORT rep;
	for(i = 0; i < g_reportInfo.reportCount; i++)
	{
		store_readData(REPORT_DATA_ADDRESS + addrOffset, (ulong*)&rep, sizeof(SUCCESS_REPORT));

		TRACE_INFO("Report:\t \tRT (%08X) - RID (%08X)\n\r", (uint)rep.readerType, (uint)rep.readerId);

		addrOffset += sizeof(SUCCESS_REPORT);
	}
#endif
}

void report_init()
{
	ulong flag = 0;

#if DBG_REPORT
	TRACE_INFO("Report:\t Init\n\r");
#endif

	store_readData(REPORT_INFO_ADDRESS, (uchar*)&flag, 4);
	if(flag != REPORT_INFO_FLAG_OK)
	{
#if DBG_REPORT
		TRACE_INFO("Report:\t OK-Flag not present, initializing data and writing OK-Flag\n\r");
#endif

		g_reportInfo.reportCount = 0;
		g_reportInfo.nextReportAddress = REPORT_DATA_ADDRESS;

		SAVE_REPORT_INFO();


		g_errorReport.magicRandValErrorCount = 0;
		g_errorReport.timeOutErrorCount = 0;

		SAVE_ERROR_REPORT();


		flag = REPORT_INFO_FLAG_OK;
		store_writeData(REPORT_INFO_ADDRESS, (uchar*)&flag, sizeof(flag));
	}

#if DBG_REPORT
	TRACE_INFO("Report:\t OK-Flag present, reading data\n\r");
#endif
	LOAD_REPORT_INFO();
	LOAD_ERROR_REPORT();


	dumpReports();
}

void report_writeSuccessReport( ulong readerType, ulong readerID, ulong flags )
{
#if DBG_REPORT
	TRACE_INFO("Report:\t Writing success report for \n\r");
	TRACE_INFO("Report:\t \t ReaderType = %08X\n\r", (uint)readerType);
	TRACE_INFO("Report:\t \t ReaderId   = %08X\n\r", (uint)readerID);
	TRACE_INFO("Report:\t \t Flags      = %08X\n\r", (uint)flags);
#endif

	SUCCESS_REPORT rep;
	rep.readerType = readerType;
	rep.readerId = readerID;
	rep.flags = flags;
	store_writeData(g_reportInfo.nextReportAddress, (uchar*)&rep, sizeof(SUCCESS_REPORT));

	LOAD_REPORT_INFO();

	g_reportInfo.nextReportAddress += sizeof(SUCCESS_REPORT);
	g_reportInfo.reportCount++;

	SAVE_REPORT_INFO();

	return;
}

void report_incrementErrorCount(ERROR_TYPE type)
{
#if DBG_REPORT
	TRACE_INFO("Report:\t Incrementing ErrorCount for %02X\n\r", type);
#endif

	LOAD_ERROR_REPORT();

	switch(type)
	{
	case MagicRandValError:
		g_errorReport.magicRandValErrorCount++;
		break;
	case TimeOutError:
		g_errorReport.timeOutErrorCount++;
		break;
	}

	SAVE_ERROR_REPORT();

	dumpReports();
}

ulong report_getReportCount()
{
	return g_reportInfo.reportCount;
}

void report_getReport(ulong index, ulong* readerId, ulong* readerType, ulong* flags)
{
	ulong addrOffset = sizeof(SUCCESS_REPORT) * index;
	SUCCESS_REPORT s;
	store_readData(REPORT_DATA_ADDRESS + addrOffset, (uchar*)&s, sizeof(SUCCESS_REPORT));
	*readerId = s.readerId;
	*readerType = s.readerType;
	*flags = s.flags;

	return;
}
