/*!
 * @file 	buffer.c
 * @brief	The buffer that is filled from the communication interfaces
 */

#include <utility/trace.h>

#include <hal/types.h>

#include <core/ringBuffer/buffer.h>

/*! @brief The maximum size of the ring buffer */
#define RING_BUFFER_SIZE		1024


/*! @brief The ring buffer */
static uchar ringBuffer[RING_BUFFER_SIZE];

/*! @brief The current read index */
static uint readIndex = 0;
/*! @brief The current write index */
static uint writeIndex = 0;


/*!
 * @brief Bytes to read
 *
 * Function returns how many bytes can be read.
 * @return	Returns the count of bytes that can be read
 */
uint buffer_bytesToRead()
{
	if(writeIndex >= readIndex)
		return writeIndex - readIndex;
	else
		return RING_BUFFER_SIZE - writeIndex + readIndex;
}

/*!
 * @brief Write into buffer
 *
 * Function to write one given byte into the ring buffer.
 * @param 	byte		The byte that should be written
 * @return	Returns if the write was successful.
 */
bool buffer_writeByte(uchar byte)
{
	if(writeIndex%RING_BUFFER_SIZE == (readIndex-1)%RING_BUFFER_SIZE)
	{
		TRACE_ERROR("Writing to full buffer...\n\r");
		return false;
	}

	ringBuffer[(writeIndex++)%RING_BUFFER_SIZE] = byte;

	return true;
}

/*!
 * @brief Read out of buffer
 *
 * Function reads one byte out uf the ringbuffer.
 * @param 	byte		Pointer to one byte where the read byte should be written at
 * @return 	Return if the read operation was successful.
 */
bool buffer_readByte(uchar* byte)
{
	if(buffer_bytesToRead() <= 0)
	{
		TRACE_ERROR("Reading from buffer while nothing to read...\n\r");
		return false;
	}

	*byte = ringBuffer[(readIndex++)%RING_BUFFER_SIZE];
	return true;
}

/*!
 * @brief Multiple read
 *
 * Function reads multiple bytes out of the ring buffer, either until the given buffer is full or until there is nothing
 * to read.
 * @param 	buffer		Pointer to a buffer where the read bytes should be written in
 * @param	buffersize	Size of the given buffer
 * @return	Returns the number of bytes that were read.
 */
uint buffer_readBytes(uchar* buffer, uint buffersize)
{
	uint readBytes = 0;
	while(buffer_bytesToRead() > 0 && readBytes < buffersize){
		buffer[readBytes] = ringBuffer[(readIndex++)%RING_BUFFER_SIZE];
		readBytes++;
	}
	return readBytes;
}
