/*
 * store.c
 *
 *  Created on: 17.06.2010
 *      Author: malte.sander
 */

#include "store.h"

#include <utility/trace.h>

#include <hal/eeprom.h>
#include <hal/flags/flags.h>


uint protocolComm_sendBytes ( uchar* buffer, uint size );



#define STORE_OK_FLAG		0x55AA2211
#define MAX_STORE_SIZE		6144


#define TRACE_INFO_ARRAY(array,len) \
{ \
	ushort i = 0; \
	for (i = 0; i < len; i++) \
	{ \
		TRACE_INFO_WP("%02x", array[i]); \
		if (i < len - 1)TRACE_INFO_WP("-"); \
	} \
	TRACE_INFO_WP("\n\r"); \
}

typedef struct
{
	ulong flag;
	ulong orderNumber;
	ulong currentSize;
} STORE_INFORMATIONS;

static STORE_INFORMATIONS g_infos;

static void INVALID_STATE(uchar info, char* str)
{
	uchar c = info;
	static uchar buffer[128];
	uchar address = 0;
	while(1){
		WDR();
		TRACE_ERROR("Store:\t %s\n\r", str);
		protocolComm_sendBytes(&c,1);

		while(1){
			WDR();
			flash_read(address, buffer, 128);
			protocolComm_sendBytes(buffer, 128);
			address += 128;
			if(address >= MAX_STORE_SIZE)
				break;
		}

	}
}


static ulong mapAddressForFlash( ulong address )
{
	// compute address for eeprom; start address = 0x0000
	return (ulong)( address + (ulong)FLASH_OFFSET + sizeof(STORE_INFORMATIONS) );
}

#if EEPROM
static ulong mapAddressForEeprom( ulong address )
{
	// compute address for eeprom; start address = 0x0000
	return (ulong)( address + (ulong)EEPROM_OFFSET + sizeof(STORE_INFORMATIONS) );
}

static bool compareFlashAgainstEeprom ( ulong currentSize )
{
	static uchar eepromBuffer[128];
	static uchar flashBuffer[128];
	uchar address = 0;
	uchar i = 0;
	uchar runs = MAX_STORE_SIZE / 128;
	for(i = 0; i < runs; i++)
	{
		if(eeprom_read(mapAddressForEeprom(address), eepromBuffer, 128) != 128)
			INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not read from eeprom!!");

		if(flash_read(mapAddressForFlash(address), flashBuffer, 128) != 128)
			INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not read from flash!!");

		uchar j = 0;
		for(j = 0; j < 128; j++)
		{
			if(eepromBuffer[j] != flashBuffer[j])
				return false;

			address++;
			if(address >= currentSize)
				return true;
		}
	}

	return true;
}

static void copyEepromToFlash ( ulong currentSize )
{
	static uchar buffer[128];
	uchar address = 0;
	uchar i = 0;
	uchar runs = MAX_STORE_SIZE / 128;
	for(i = 0; i < runs; i++)
	{
		if(eeprom_read(mapAddressForEeprom(address), buffer, 128) != 128)
			INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not read from eeprom!!");
		if(flash_write(mapAddressForFlash(address), buffer, 128) != 128)
			INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not write to flash!!");

		address += 128;

		if(address >= currentSize)
			return;
	}
}

static bool getInformationBlockEeprom ( STORE_INFORMATIONS* infos )
{
	return (eeprom_read(EEPROM_OFFSET, (uchar*)infos, sizeof(STORE_INFORMATIONS)) == sizeof(STORE_INFORMATIONS));
}
#endif

static bool getInformationBlockFlash ( STORE_INFORMATIONS* infos )
{
	return (flash_read(FLASH_OFFSET, (uchar*)infos, sizeof(STORE_INFORMATIONS)) == sizeof(STORE_INFORMATIONS));
}

static bool writeInformationBlocks ( STORE_INFORMATIONS* infos )
{
#if DBG_STORE
	TRACE_INFO("Store:\t Writing information block:\n\r");
	TRACE_INFO("Store:\t \tFlag          = %08X\n\r", (uint)infos->flag);
	TRACE_INFO("Store:\t \tSerialNumber  = %08X\n\r", (uint)infos->orderNumber);
	TRACE_INFO("Store:\t \tCurrentSize   = %d Bytes\n\r", (uint)infos->currentSize);
#endif

	if(flash_write(FLASH_OFFSET, (uchar*)infos, sizeof(STORE_INFORMATIONS)) != sizeof(STORE_INFORMATIONS))
		return false;

#if EEPROM
	if(eeprom_write(EEPROM_OFFSET, (uchar*)infos, sizeof(STORE_INFORMATIONS)) != sizeof(STORE_INFORMATIONS))
		return false;
#endif

	return true;
}

ulong store_orderNumber()
{
	return g_infos.orderNumber;
}

void store_init( ulong orderNumber )
{
	flash_initDriver();
#if EEPROM
	eeprom_init();
#endif

	if(!getInformationBlockFlash(&g_infos))
		INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not read from flash!!");

	if(g_infos.flag == STORE_OK_FLAG)
	{
#if EEPROM

		STORE_INFORMATIONS eepromInfos;
		if(!getInformationBlockEeprom(&eepromInfos))
			INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not read from eeprom!!");

		if(eepromInfos.flag != g_infos.flag ||
		   eepromInfos.orderNumber != g_infos.orderNumber ||
		   eepromInfos.currentSize != g_infos.currentSize ||
		   compareFlashAgainstEeprom(g_infos.currentSize) == false)
		{
			INVALID_STATE(STORE_INCONSISTANT_PROBLEM, "Inconsistant Memory!!");
		}

#endif
#if DBG_STORE
		TRACE_INFO("Store:\t Flash informations OK\n\r");
#endif
		return;
	}
#if EEPROM
	else
	{
		if(!getInformationBlockEeprom(&g_infos))
			INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not read from eeprom!!");

		if(g_infos.flag == STORE_OK_FLAG)
		{
#if DBG_STORE
			TRACE_INFO("Store:\t Eeeprom informations OK -> Copying to Flash\n\r");
#endif
			copyEepromToFlash(g_infos.currentSize);
			writeInformationBlocks(&g_infos);
		}
		else
		{
#if DBG_STORE
			TRACE_INFO("Store:\t Writing initial informations.\n\r");
#endif
			g_infos.flag = STORE_OK_FLAG;
			g_infos.currentSize = 0;
			g_infos.orderNumber = orderNumber;
			writeInformationBlocks(&g_infos);
		}
	}
#else
	else
	{
#if DBG_STORE
		TRACE_INFO("Store:\t Writing initial informations.\n\r");
#endif
		g_infos.flag = STORE_OK_FLAG;
		g_infos.currentSize = 0;
		g_infos.orderNumber = orderNumber;
		writeInformationBlocks(&g_infos);
	}
#endif
}



void store_writeData( ulong address, uchar* data, ulong length )
{
#if DBG_STORE
	TRACE_INFO("Store:\t Write %d bytes to address %08X\n\r", (uint)length, (uint)address);
#endif

	if(length != flash_write(mapAddressForFlash(address), data, length))
		INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not write to flash!!");

#if EEPROM
	if(length != eeprom_write(mapAddressForEeprom(address), data, length))
		INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not write to flash!!");
#endif

	if( (address + length) > g_infos.currentSize )
	{
#if DBG_STORE
		TRACE_INFO("Store:\t Updating size from %d to %d Bytes\n\r", (uint)g_infos.currentSize, (uint)(address + length) );
#endif
		g_infos.currentSize = address + length;
		writeInformationBlocks(&g_infos);
	}

	return;
}

void store_readData( ulong address, uchar* data, ulong length )
{
#if DBG_STORE
	TRACE_INFO("Store:\t Read %d bytes from address %08X\n\r", (uint)length, (uint)address);
#endif

	if(length != flash_read(mapAddressForFlash(address), data, length))
		INVALID_STATE(STORE_ACCESS_PROBLEM, "Can not read from flash!!");

	return;
}
