/*!
 * @file	commCDC.c
 * @brief	Implementation of the USB CDC interface
 */
#include <board.h>
#include <pio/pio.h>
#include <pio/pio_it.h>
#include <aic/aic.h>
#include <usart/usart.h>
#include <utility/trace.h>
#include <usb/device/cdc-serial/CDCDSerialDriver.h>
#include <usb/device/cdc-serial/CDCDSerialDriverDescriptors.h>

#include <hal/types.h>

#include <core/ringBuffer/buffer.h>

#include <def.h>


/*! @brief Size in bytes of the buffer used for reading data from the USB */
#define DATABUFFERSIZE \
    BOARD_USB_ENDPOINTS_MAXPACKETSIZE(CDCDSerialDriverDescriptors_DATAIN)



/*! @brief Buffer for storing incoming USB data */
static uchar usbBuffer[DATABUFFERSIZE];


/*! @brief A rename definition */
#define VBUS_CONFIGURE()    USBD_Connect()



/*! @brief Callback invoked when data has been received on the USB */
static void usbDataReceived(uint unused,
                     uchar status,
                     uint received,
                     uint remaining)
{
	uint i = 0;
#if DBG_USBCDC
	TRACE_DEBUG("\n\n\rUSB-CDC:\t UsbDataReceived\n\r");
#endif
	// Check that data has been received successfully
    if (status == USBD_STATUS_SUCCESS) {
#if DBG_USBCDC
    	TRACE_DEBUG("USB-CDC:\t UsbDataReceived: %u bytes received\n\r", received);
#endif

    	for(i = 0; i < received; i++){
    		buffer_writeByte(usbBuffer[i]);
    	}

        // Check if bytes have been discarded
        if ((received == DATABUFFERSIZE) && (remaining > 0)) {

#if DBG_USBCDC
            TRACE_WARNING("USB-CDC:\t UsbDataReceived: %u bytes discarded\n\r", remaining);
#endif
       }
    }
    else {

#if DBG_USBCDC
        TRACE_WARNING("USB-CDC:\t UsbDataReceived: Transfer error\n\r");
#endif
    }
}

/*! @brief Callback invoked when data has been written to the USB */
static void usbDataWritten(uint unused,
                     uchar status,
                     uint transferred,
                     uint remaining)
{
#if DBG_USBCDC
	TRACE_DEBUG("\n\n\rUSB-CDC:\t UsbDataWritten\n\r");
#endif

    // Check that data has been received successfully
    if (status == USBD_STATUS_SUCCESS) {
#if DBG_USBCDC
    	TRACE_DEBUG("USB-CDC:\t UsbDataWritten: %u bytes successfully written\n\r", transferred);
#endif
    	// Check if bytes have been discarded
        if ((transferred == DATABUFFERSIZE) && (remaining > 0)) {

#if DBG_USBCDC
            TRACE_WARNING("USB-CDC:\t UsbDataWritten: %u bytes discarded\n\r", remaining);
#endif
        }
    }
	else {
#if DBG_USBCDC
		TRACE_WARNING("USB-CDC:\t UsbDataWritten: Transfer error\n\r");
#endif
	}

}


/*!
 * @brief Init function for usb cdc
 *
 * Function initializes the usb cdc interface.
 */
void cdc_init(void)
{
#if DBG_USBCDC
	TRACE_WARNING("USB-CDC:\t Configuring CDC...\n\r");
#endif

    // If they are present, configure Vbus & Wake-up pins
    PIO_InitializeInterrupts(0);

	// BOT driver initialization
	CDCDSerialDriver_Initialize();

    // connect if needed
    VBUS_CONFIGURE();

#if DBG_USBCDC
	TRACE_WARNING("USB-CDC:\t Waiting for configured by host...\n\r");
#endif

	// TODO: remove to get usb together with serial
	// Wait until connected
	while (USBD_GetState() < USBD_STATE_CONFIGURED);

    // Connect pull-up, wait for configuration
    USBD_Connect();

#if DBG_USBCDC
	TRACE_WARNING("USB-CDC:\t Configured from host...\n\r");
#endif
}


/*!
 * @brief Process function for usb cdc
 *
 * Process function for usb cdc that should be called periodically.
 */
void cdc_process(void)
{
	//trace_LOG(trace_WARNING, "Processing...\n\r");
    // Start receiving data on the USB
    CDCDSerialDriver_Read(usbBuffer,
                          DATABUFFERSIZE,
                          (TransferCallback) usbDataReceived,
                          0);
}


/*!
 * @brief Write function for the cdc
 *
 * Function writes the given bytes to the cdc.
 * @param	buffer		Pointer to the data that should be written
 * @param	size		Size of the data
 * @return	Returns the count of written bytes.
 */
uint cdc_write(uchar* buffer, uint size)
{
	CDCDSerialDriver_Write(buffer, size, (TransferCallback) usbDataWritten, 0);
	return size;
}
