/*
 * eeprom.c
 *
 *  Created on: 03.08.2009
 *      Author: stefan.detter
 */

#include "eeprom.h"

#include <aic/aic.h>
#include <pio/pio.h>
#include <twi/twi.h>
#include <drivers/twi/twid.h>
#include <utility/trace.h>

#define MEMORY_SIZE		8192

/// TWI clock frequency in Hz.
#define TWCK            400000

/// Slave address of AT24C chips.
#define AT24C_ADDRESS   0x50

/// Page size of an AT24C64C chip (in bytes)
#define PAGE_SIZE       32

/// TWI driver instance.
static Twid twid;


/*!
 * @brief Debug macro to print array
 *
 * The Macro prints out the given array.
 * @param 	array	The array that should be printed
 * @param	len		The length of the array
 */
#define TRACE_INFO_ARRAY(array,len) \
{ \
	ushort i = 0; \
	for (i = 0; i < len; i++) \
	{ \
		TRACE_INFO_WP("%02x", array[i]); \
		if (i < len - 1)TRACE_INFO_WP("-"); \
	} \
	TRACE_INFO_WP("\n\r"); \
}


static void wait(uint ticks)
{
	volatile uint i;
	volatile uint j;
	for(i = 0; i < ticks; i++)
		j = i;
}



void eeprom_init ( void )
{
#if DBG_EEPROM
	TRACE_INFO("Eeprom:\t Init\n\r");
#endif

	/// Pio pins to configure.
	Pin pins[] = {PINS_TWI};
    PIO_Configure(pins, PIO_LISTSIZE(pins));

    // Configure TWI
    // In IRQ mode: to avoid problems, the priority of the TWI IRQ must be max.
    // In polling mode: try to disable all IRQs if possible.
    // (in this example it does not matter, there is only the TWI IRQ active)
    AT91C_BASE_PMC->PMC_PCER = 1 << AT91C_ID_TWI;
    TWI_ConfigureMaster(AT91C_BASE_TWI, TWCK, BOARD_MCK);
    TWID_Initialize(&twid, AT91C_BASE_TWI);
}

ushort eeprom_read( ushort address, uchar* buffer, ushort size )
{
	if(address + size > MEMORY_SIZE){
		TRACE_ERROR("Eeprom:\t READ - Trying to read from too high address\n\r");
		return 0;
	}

#if DBG_EEPROM
	TRACE_INFO("Eeprom:\t READ - Reading from address %d the bytes %d \n\r", address, size);
#endif

	ushort page_end = 0;
	while(page_end <= address)
		page_end += PAGE_SIZE;

	int i = 0;
	for(i = 0; i < size; i++)
		buffer[i] = 0;

	ushort temp_address = address;
	ushort bytes_to_read = size;
	ushort temp_size = 0;
	uchar* temp_buffer = buffer;
	do
	{
		wait(10000);

		if( (temp_address + bytes_to_read) > page_end)
			temp_size = page_end - temp_address;
		else
			temp_size = bytes_to_read;

#if DBG_EEPROM
		TRACE_INFO("Eeprom:\t READ - Reading from address %d the count %d into buffer[%d]\n\r", temp_address, temp_size, (uint)(temp_buffer - buffer));
#endif
		// Read back data
		if(TWID_Read(&twid, AT24C_ADDRESS, temp_address, 2, temp_buffer, temp_size, 0) != 0)
			return 0;

#if DBG_EEPROM
		TRACE_INFO("Eeprom:\t READ - ");
		TRACE_INFO_ARRAY(temp_buffer, temp_size);
#endif

		temp_address = page_end;
		page_end += PAGE_SIZE;
		temp_buffer += temp_size;
		bytes_to_read -= temp_size;

	} while (bytes_to_read > 0);

    return size;
}

ushort eeprom_write( ushort address, uchar* buffer, ushort size )
{
	if(address + size > MEMORY_SIZE){
		TRACE_ERROR("Eeprom:\t WRITE - Trying to write to too high address\n\r");
		return 0;
	}

#if DBG_EEPROM
	TRACE_INFO("Eeprom:\t WRITE - Writing to address %d the bytes %d \n\r", address, size);
#endif

	ushort page_end = 0;
	while(page_end <= address)
		page_end += PAGE_SIZE;

	ushort temp_address = address;
	ushort bytes_to_write = size;
	ushort temp_size = 0;
	uchar* temp_buffer = buffer;
	do
	{
		wait(10000);

		if( (temp_address + bytes_to_write) > page_end)
			temp_size = page_end - temp_address;
		else
			temp_size = bytes_to_write;

#if DBG_EEPROM
		TRACE_INFO("Eeprom:\t WRITE - Writting to address %d the count %d from buffer[%d]\n\r", temp_address, temp_size, (uint)(temp_buffer - buffer));
#endif

		// Read back data
		if(TWID_Write(&twid, AT24C_ADDRESS, temp_address, 2, temp_buffer, temp_size, 0) != 0)
			return 0;

		temp_address = page_end;
		page_end += PAGE_SIZE;
		temp_buffer += temp_size;
		bytes_to_write -= temp_size;

	} while (bytes_to_write > 0);

    return size;
}
