/*!
 * @file	flags.c
 * @brief	Helper functions to read and write the flash
 */

#include "flags.h"

#include <board.h>
#include <utility/trace.h>
#include <memories/flash/flashd.h>

/*!
 * @brief Function to write data to the flash memory
 *
 * This function writes the given value to the given address in the flash memory.
 * @param	address			The address in the flash memory
 * @param	value			The value that should be written
 * @returns	Return the success of the function
 */
bool flags_setFlashValue ( ulong address, ulong value )
{
	ulong* flashValue = (ulong*)address;
	if(*flashValue == value)
		return true;

	ulong interrupt_register = AT91C_BASE_AIC->AIC_IMR;
	AT91C_BASE_AIC->AIC_IDCR = interrupt_register;
	AT91C_BASE_AIC->AIC_ICCR = interrupt_register;

	while(AT91C_BASE_AIC->AIC_IMR != 0x00000000){
//		TRACE_ERROR("RHImpl:\t Waiting for disable complete...\n\r");
	}

	uchar ret = FLASHD_Write((uint)address, (uint*)&value, sizeof(value));

	AT91C_BASE_AIC->AIC_IECR = interrupt_register;
//	TRACE_ERROR("RHImpl:\t Reenable the interrupts\n\r");

	if(ret == 0)
		return true;
	else
		return false;
}

/*!
 * @brief Function to get data from the flash memory
 *
 * This function reads a value from the given address from the flash memory.
 * @param	address			The address in the flash memory
 * @returns	Return the read value
 */
ulong flags_getFlashValue ( ulong address )
{
	ulong* ptr = (ulong*)address;
	return *ptr;
}


ulong flash_read( ulong address, uchar* buffer, ulong size )
{
#if DBG_FLASH
	TRACE_INFO("Flash:\t Read %d bytes from address %d\n\r", (uint)size, (uint)address);
#endif

	ulong i = 0;
	uchar* ptr = (uchar*)address;
	for(i = 0; i < size; i++)
		buffer[i] = ptr[i];

	return size;
}

ulong flash_write( ulong address, uchar* buffer, ulong size )
{
#if DBG_FLASH
	TRACE_INFO("Flash:\t Write %d bytes to address %d\n\r", (uint)size, (uint)address);
#endif

	ulong interrupt_register = AT91C_BASE_AIC->AIC_IMR;
	AT91C_BASE_AIC->AIC_IDCR = interrupt_register;
	AT91C_BASE_AIC->AIC_ICCR = interrupt_register;

	while(AT91C_BASE_AIC->AIC_IMR != 0x00000000){
//		TRACE_ERROR("RHImpl:\t Waiting for disable complete...\n\r");
	}

	uchar ret = FLASHD_Write((uint)address, buffer, size);

	AT91C_BASE_AIC->AIC_IECR = interrupt_register;

	if(ret == 0)
		return size;
	return 0;
}

/*! @brief Function to initialize the driver for reading and writing flags to persistant memory */
void flash_initDriver ( )
{
	FLASHD_Initialize(BOARD_MCK);
}
