/*!
* @file main.c
* @mainpage License Dongle
*
* <b>project description: </b><br><br>
* The License Dongle Project is a hardware license manager. It needs USB / CDC communication to a Host PC. <br><br>
* 1. USB/CDC (115200 Baud) <br>
*
* @author malte.sander
*/

#include <board.h>
#include <utility/trace.h>
#include <stdio.h>

#include <def.h>
#include <hal/types.h>

#include <core/protocol/readerHostProtocol.h>
#include <core/protocol/readerHostProtocolImpl.h>
#include <core/protocol/licenseProtocol.h>
#include <core/store/store.h>
#include <core/licenseMgr/licenseMgr.h>
#include <core/random/random.h>
#include <core/report/report.h>

#include <hal/flags/flags.h>

#include <memories/flash/flashd.h>

//------------------------------------------------------------------------------
//  	Global Variables
//------------------------------------------------------------------------------

/*! @brief increased every main while(1) cycle */
uint g_timeoutCounter = 0;

//------------------------------------------------------------------------------
//  	Main
//------------------------------------------------------------------------------

int main(void)
{
	// DBGU configuration
	TRACE_CONFIGURE(DBGU_STANDARD, 115200, BOARD_MCK);
	TRACE_INFO_WP("\n\n\r-- License Dongle Project with Softpack Version %s --\n\r", SOFTPACK_VERSION);
	TRACE_INFO_WP("Board : %s, Chip ID : 0x%08X\n\r", BOARD_NAME, AT91C_BASE_DBGU->DBGU_CIDR);
	TRACE_INFO_WP("Compiled: %s %s \n\r", __DATE__, __TIME__);
	TRACE_INFO_WP("The board runs with %d Hz \n\r\n\r\n\r", BOARD_MCK);

	// configurate cdc communication
	protocol_init( );

	store_init(ORDER_NUMBER);

	// init license protocol
	licProtocol_init( );

	// init licensemanager
	licmgr_init( );

	// init reports
	report_init( );

#if SECURITY_BIT == 1
	// if not set already, set the security bit to avoid extern access
	if(!FLASHD_IsSecurityBitSet())
	{
		uchar ret = FLASHD_SetSecurityBit();
		TRACE_INFO("Set Securitiy Bit: %s", ret == 0 ? "successfull!\n\r" : "error!\n\r");
	}
#endif

	// unlock flash
	if (FLASHD_IsLocked((uint) FLASH_START_ADDRESS, (uint) (FLASH_START_ADDRESS	+ 0x40000)) != 0)
	{
		if (FLASHD_Unlock((uint) FLASH_START_ADDRESS, (uint) (FLASH_START_ADDRESS + 0x40000), 0, 0) != 0)
		{
			TRACE_ERROR("Unlocking Flash failed! --> while(1); \n\r");
			while(1);
		}
	}

	// Main loop
	while (1)
	{
		protocol_process();

		if(g_timeoutCounter > 130000) // ca. 100ms maybe better testet and adapted
		{
			if(!licProtocol_atReset())
			{
				TRACE_INFO("Main:\t TIME_OUT...\n\r");
				report_incrementErrorCount(TimeOutError);
				protocol_sendLicenseResult(false);
			}
			licProtocol_resetState();
			g_timeoutCounter = 0;
		}

		g_rand_seed++;
		g_timeoutCounter++;
	}
}

