﻿/*
 * Copyright (c) 2008-2018, RF-Embedded GmbH
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CSrfeReaderInterface.protocol;

namespace CSrfeReaderInterface.protocol
{
    public class CSrfePURprotocolHandler : CSrfeProtocolHandler
    {
        /// <summary>
        /// Constructs an instnace of the protocol handler for the given device.
        /// </summary>
        /// <param name="device">Device to which the reader is connected</param>
        public CSrfePURprotocolHandler(device.IProtocolDeviceInterface device) : base(device)
        {
        }

        /// <summary>
        /// Typedef for lock mode parameter
        /// </summary>
        public enum eRFE_LOCK_MODE
        {
            /// <summary>Unlocks mem space</summary>
            UNLOCK = 0x00, 
            /// <summary>Locks mem space</summary>
            LOCK = 0x01,
            /// <summary>Locks mem space permanent</summary>
            PERMALOCK = 0x02, 
            /// <summary>Locks mem space with passw and permanent</summary>
            LOCK_AND_PERMALOCK = 0x03,
        };

        /// <summary>
        /// Typedef for lock mem space parameter
        /// </summary>
        public enum eRFE_LOCK_MEMORY_SPACE
        {
            /// <summary>Kill password</summary>
            KILL_PASSWORD = 0x00, 
            /// <summary>Access password</summary>
            ACCESS_PASSWORD = 0x01, 
            /// <summary>EPC mem bank</summary>
            EPC = 0x02, 
            /// <summary>TID mem bank</summary>
            TID = 0x03,
            /// <summary>USER mem bank</summary>
            USER = 0x04,
        };


        #region Settings

            /// <summary>
            /// Tries to get the post detect READ settings
            /// </summary>
            /// <param name="on">Is post detect READ on?</param>
            /// <param name="memBank">Target Memory Bank</param>
            /// <param name="address">Target Address</param>
            /// <param name="size">Target Size</param>
            /// <param name="password">Password</param>
            /// <returns>Success of the operation</returns>
            public bool getPostDetectReadParams(out bool on, out byte memBank, out ushort address, out byte size, out byte[] password)
            {
                on = false;
                memBank = 0;
                address = 0;
                size = 0;
                password = new byte[0];

                byte[] buffer;
                if (!getParam(0x0006, out buffer))
                    return false;

                if (buffer.Count() != 9)
                    return false;

                on = (buffer[0] != 0);
                memBank = buffer[1];
                address = 0;
                address += (ushort)(((ushort)buffer[2]) << 8);
                address += (ushort)(((ushort)buffer[3]) << 0);
                size = buffer[4];
                password = new byte[4];
                password[0] = buffer[5];
                password[1] = buffer[6];
                password[2] = buffer[7];
                password[3] = buffer[8];

                return true;
            }

            /// <summary>
            /// Tries to set the post detect READ settings
            /// </summary>
            /// <param name="on">Should post detect READ be on?</param>
            /// <param name="memBank">Target Memory Bank</param>
            /// <param name="address">Target Address</param>
            /// <param name="size">Target Size</param>
            /// <param name="password">Password</param>
            /// <returns>Success of the operation</returns>
            public bool setPostDetectReadParams(bool on, byte memBank, ushort address, byte size, byte[] password)
            {
                byte[] buffer = new byte[9];
                buffer[0] = (byte)((on)?1:0);
                buffer[1] = memBank;
                buffer[2] = (byte)(address >> 8);
                buffer[3] = (byte)(address >> 0);
                buffer[4] = size;
                buffer[5] = password[0];
                buffer[6] = password[1];
                buffer[7] = password[2];
                buffer[8] = password[3];
            
                return setParam(0x0006, buffer);
            }


            /// <summary>
            /// Tries to get the used read block size
            /// </summary>
            /// <param name="wordCount">The count of words, that are read with one gen2 read operation</param>
            /// <returns>Success of the operation</returns>
            public bool getReadBlockSize(out byte wordCount)
            {
                wordCount = 0;

                byte[] buffer;
                if (!getParam(0x002C, out buffer))
                    return false;

                if (buffer.Count() != 1)
                    return false;

                wordCount = buffer[0];

                return true;
            }

            /// <summary>
            /// Tries to set the used read block size
            /// </summary>
            /// <param name="wordCount">The count of words, that are read with one gen2 read operation</param>
            /// <returns>Success of the operation</returns>
            public bool setReadBlockSize(byte wordCount)
            {
                byte[] buffer = new byte[1];
                buffer[0] = wordCount;

                return setParam(0x002C, buffer);
            }

        #endregion Settings

        #region Tag-Functions

            /// <summary>
            /// Tries to execute a block-write command
            /// </summary>
            /// <param name="epc">The known EPC of the tag</param>
            /// <param name="mem_bank">The retrieved handle</param>
            /// <param name="address">The known EPC of the tag</param>
            /// <param name="passwd">The retrieved handle</param>
            /// <param name="blockSize">The block size of BlockWrite that should be used in words</param>
            /// <param name="data">The retrieved handle</param>
            /// <returns>Success of the operation</returns>
            public bool blockWriteToTag(byte[] epc, byte mem_bank, ushort address, byte[] passwd, byte blockSize, byte[] data)
            {
                if (data.Count() > 220)
                {
                    Global.trc(3, "BlockWrite To Tag - NOK - Data");
                    return false;
                }

                if (passwd.Length != 4)
                {
                    Global.trc(3, "BlockWrite To Tag - NOK - Data");
                    return false;
                }

                byte[] payload;
                List<byte> payloadList = new List<byte>();
                payloadList.Add((byte)epc.Count());
                payloadList.AddRange(epc);

                payloadList.Add((byte)mem_bank);
                payloadList.Add((byte)(address >> 8));
                payloadList.Add((byte)address);
                payloadList.AddRange(passwd);
                payloadList.Add((byte)blockSize);
                payloadList.Add((byte)data.Count());
                payloadList.AddRange(data);

                payload = payloadList.ToArray();

                byte[] resultData;

                if (!customTagCommand((byte)0x03, payload, out resultData))
                    return false;

                return true;
            }

            /// <summary>
            /// Retrieves the handle of a tag
            /// </summary>
            /// <param name="epc">The known EPC of the tag</param>
            /// <param name="handle">The retrieved handle</param>
            /// <returns>Success of the operation</returns>
            public bool getTagHandle(byte[] epc, out byte[] handle)
            {
                handle = new byte[0];

                byte[] payload;
                List<byte> payloadList = new List<byte>();
                payloadList.Add((byte)epc.Count());
                payloadList.AddRange(epc);
                payload = payloadList.ToArray();

                byte[] resultData;

                if (!customTagCommand((byte)0x04, payload, out resultData))
                    return false;

                if (resultData.Count() < 2)
                    return false;

                handle = new byte[2];
                Array.Copy(resultData, 0, handle, 0, 2);

                return true;
            }

            /// <summary>
            /// Reads data direct from the handle of a tag
            /// </summary>
            /// <param name="handle">The handle of the tag</param>
            /// <param name="mem_bank">The mem bank to read from</param>
            /// <param name="address">The mem address to read from</param>
            /// <param name="passwd">The passwd to be used</param>
            /// <param name="count">The count of data to read from</param>
            /// <param name="data">The read data</param>
            /// <returns>Success of the operation</returns>
            public bool readFromHandle(byte[] handle, byte mem_bank, ushort address, byte[] passwd, byte count, out byte[] data)
            {
                data = new byte[0];

                if (handle.Count() != 2)
                {
                    Global.trc(3, "Read From Handle - NOK - Data");
                    return false;
                }

                if (passwd.Length != 4)
                {
                    Global.trc(3, "Read From Handle - NOK - Data");
                    return false;
                }

                byte[] payload;
                List<byte> payloadList = new List<byte>();
                payloadList.AddRange(handle);

                payloadList.Add((byte)mem_bank);
                payloadList.Add((byte)(address >> 8));
                payloadList.Add((byte)address);
                payloadList.AddRange(passwd);

                payloadList.Add(count);
                payload = payloadList.ToArray();

                byte[] resultData;

                if (!customTagCommand((byte)0x05, payload, out resultData))
                    return false;

                data = new byte[resultData.Length - 1];
                Array.Copy(resultData, 1, data, 0, resultData.Length - 1);

                return true;
            }

            /// <summary>
            /// Writes data direct to the handle of a tag
            /// </summary>
            /// <param name="handle">The handle of the tag</param>
            /// <param name="mem_bank">The mem bank to write to</param>
            /// <param name="address">The mem address to write to</param>
            /// <param name="passwd">The passwd to be used</param>
            /// <param name="data">The data to be written</param>
            /// <returns>Success of the operation</returns>
            public bool writeToHandle(byte[] handle, byte mem_bank, ushort address, byte[] passwd, byte[] data)
            {
                if (handle.Count() != 2)
                {
                    Global.trc(3, "Write To Handle - NOK - Data");
                    return false;
                }
            
                if (data.Count() > 220)
                {
                    Global.trc(3, "Write To Handle - NOK - Data");
                    return false;
                }

                if (passwd.Length != 4)
                {
                    Global.trc(3, "Write To Handle - NOK - Data");
                    return false;
                }

                byte[] payload;
                List<byte> payloadList = new List<byte>();
                payloadList.AddRange(handle);

                payloadList.Add((byte)mem_bank);
                payloadList.Add((byte)(address >> 8));
                payloadList.Add((byte)address);
                payloadList.AddRange(passwd);
                payloadList.Add((byte)data.Count());
                payloadList.AddRange(data);
                payload = payloadList.ToArray();

                byte[] resultData;

                if (!customTagCommand((byte)0x06, payload, out resultData))
                    return false;

                return true;
            }
        
            /// <summary>
            /// BlockWrites data direct to the handle of a tag
            /// </summary>
            /// <param name="handle">The handle of the tag</param>
            /// <param name="mem_bank">The mem bank to write to</param>
            /// <param name="address">The mem address to write to</param>
            /// <param name="passwd">The passwd to be used</param>
            /// <param name="blockSize">The block size of BlockWrite that should be used in words</param>
            /// <param name="data">The data to be written</param>
            /// <returns>Success of the operation</returns>
            public bool blockWriteToHandle(byte[] handle, byte mem_bank, ushort address, byte[] passwd, byte blockSize, byte[] data)
            {
                if (handle.Count() != 2)
                {
                    Global.trc(3, "BlockWrite To Handle - NOK - Data");
                    return false;
                }

                if (data.Count() > 220)
                {
                    Global.trc(3, "BlockWrite To Handle - NOK - Data");
                    return false;
                }

                if (passwd.Length != 4)
                {
                    Global.trc(3, "BlockWrite To Handle - NOK - Data");
                    return false;
                }

                byte[] payload;
                List<byte> payloadList = new List<byte>();
                payloadList.AddRange(handle);

                payloadList.Add((byte)mem_bank);
                payloadList.Add((byte)(address >> 8));
                payloadList.Add((byte)address);
                payloadList.AddRange(passwd);
                payloadList.Add((byte)blockSize);
                payloadList.Add((byte)data.Count());
                payloadList.AddRange(data);
                payload = payloadList.ToArray();

                byte[] resultData;

                if (!customTagCommand((byte)0x08, payload, out resultData))
                    return false;

                return true;
            }


            /// <summary>
            /// Sends a custom tag command to the tag
            /// </summary>
            /// <param name="handle">The handle of the tag</param>
            /// <param name="command">The two byte command to be sent</param>
            /// <param name="passwd">The passwd to be used</param>
            /// <param name="txBitCount">The count of bits to be sent</param>
            /// <param name="txBits">The bits to be sent</param>
            /// <param name="headerBit">Flag to signalize, if header bit was received</param>
            /// <param name="estimatedRxCount">The estimated count of rx bits</param>
            /// <param name="rxBytes">The bytes received from the tag</param>
            /// <returns>Success of the operation</returns>
            public bool customGen2Command(byte[] handle, byte[] command, byte[] passwd, byte txBitCount, byte[] txBits, byte estimatedRxCount, out bool headerBit, out byte[] rxBytes)
            {
                List<byte> additionalInfo = new List<byte>();
                return customGen2Command(handle, command, passwd, txBitCount, txBits, estimatedRxCount, additionalInfo.ToArray(), out headerBit, out rxBytes);
            }

            /// <summary>
            /// Sends a custom tag command to the tag
            /// </summary>
            /// <param name="handle">The handle of the tag</param>
            /// <param name="command">The two byte command to be sent</param>
            /// <param name="passwd">The passwd to be used</param>
            /// <param name="txBitCount">The count of bits to be sent</param>
            /// <param name="txBits">The bits to be sent</param>
            /// <param name="headerBit">Flag to signalize, if header bit was received</param>
            /// <param name="estimatedRxCount">The estimated count of rx bits</param>
            /// <param name="estimatedTagProcessingTime">The estimated prossing time in ms of the tag, before replying</param>
            /// <param name="rxBytes">The bytes received from the tag</param>
            /// <returns>Success of the operation</returns>
            public bool customGen2Command(byte[] handle, byte[] command, byte[] passwd, byte txBitCount, byte[] txBits, byte estimatedRxCount, byte estimatedTagProcessingTime, out bool headerBit, out byte[] rxBytes)
            {
                List<byte> additionalInfo = new List<byte>();
                additionalInfo.Add((byte)estimatedTagProcessingTime);
                return customGen2Command(handle, command, passwd, txBitCount, txBits, estimatedRxCount, additionalInfo.ToArray(), out headerBit, out rxBytes);
            }

            /// <summary>
            /// Sends a custom tag command to the tag
            /// </summary>
            /// <param name="handle">The handle of the tag</param>
            /// <param name="command">The two byte command to be sent</param>
            /// <param name="passwd">The passwd to be used</param>
            /// <param name="txBitCount">The count of bits to be sent</param>
            /// <param name="txBits">The bits to be sent</param>
            /// <param name="headerBit">Flag to signalize, if header bit was received</param>
            /// <param name="estimatedRxCount">The estimated count of rx bits</param>
            /// <param name="additionalInfo">Additional Info like estimated RX count or tag processing time</param>
            /// <param name="rxBytes">The bytes received from the tag</param>
            /// <returns>Success of the operation</returns>
            private bool customGen2Command(byte[] handle, byte[] command, byte[] passwd, byte txBitCount, byte[] txBits, byte estimatedRxCount, byte[] additionalInfo, out bool headerBit, out byte[] rxBytes)
            {
                rxBytes = new byte[0];
                headerBit = true;

                if (handle.Count() != 2)
                {
                    Global.trc(3, "Custom Tag Command Handle - NOK - Data");
                    return false;
                }

                if (command.Length != 2 || passwd.Length != 4)
                {
                    Global.trc(3, "Custom Tag Command Handle - NOK - Data");
                    return false;
                }

                if (passwd.Count() != 4)
                {
                    Global.trc(3, "Custom Tag Command Handle - NOK - Data");
                    return false;
                }

                byte txByteCount = (byte)(txBitCount >> 3);
                if ((txBitCount % 8) > 0)
                    txByteCount++;

                if (txByteCount != txBits.Length)
                {
                    Global.trc(3, "Custom Tag Command Handle - NOK - Data");
                    return false;
                }


                byte[] payload;
                List<byte> payloadList = new List<byte>();
                payloadList.AddRange(handle);
                payloadList.AddRange(command);
                payloadList.AddRange(passwd);
                payloadList.Add((byte)txBitCount);
                payloadList.AddRange(txBits);
                payloadList.Add((byte)estimatedRxCount);
                payloadList.AddRange(additionalInfo);

                payload = payloadList.ToArray();

                byte[] resultData;

                if (!customTagCommand((byte)0x07, payload, out resultData))
                    return false;

                headerBit = resultData[0] != 0;

                rxBytes = new byte[resultData.Length - 2];
                Array.Copy(resultData, 2, rxBytes, 0, resultData.Length - 2);

                return true;
            }


            /// <summary>
            /// Sends a direct command to the tag
            /// </summary>
            /// <param name="sendwCRC">Specifies if the data should be sent with CRC</param>
            /// <param name="sendwTRCAL">Specifies if the data should be sent with TRCal Preamble</param>
            /// <param name="txBitCount">The count of bits to be sent</param>
            /// <param name="txBits">The bits to be sent</param>
            /// <param name="headerBit">Flag to signalize, if header bit was received</param>
            /// <param name="estimatedRxCount">The estimated count of rx bits</param>
            /// <param name="rxBytes">The bytes received from the tag</param>
            /// <returns>Success of the operation</returns>
            public bool directCommand(byte sendwCRC, byte sendwTRCAL, ushort txBitCount, byte[] txBits, ushort estimatedRxCount, out bool headerBit, out byte[] rxBytes)
            {
                List<byte> additionalInfo = new List<byte>();
                return directCommand(sendwCRC, sendwTRCAL, txBitCount, txBits, estimatedRxCount, additionalInfo.ToArray(), out headerBit, out rxBytes);
            }

            /// <summary>
            /// Sends a direct command to the tag
            /// </summary>
            /// <param name="sendwCRC">Specifies if the data should be sent with CRC</param>
            /// <param name="txBitCount">The count of bits to be sent</param>
            /// <param name="txBits">The bits to be sent</param>
            /// <param name="headerBit">Flag to signalize, if header bit was received</param>
            /// <param name="estimatedRxCount">The estimated count of rx bits</param>
            /// <param name="estimatedTagProcessingTime">The estimated prossing time in ms of the tag, before replying</param>
            /// <param name="rxBytes">The bytes received from the tag</param>
            /// <returns>Success of the operation</returns>
            public bool directCommand(byte sendwCRC, byte sendwTRCAL, ushort txBitCount, byte[] txBits, byte estimatedRxCount, ushort estimatedTagProcessingTime, out bool headerBit, out byte[] rxBytes)
            {
                List<byte> additionalInfo = new List<byte>();
                additionalInfo.Add((byte)estimatedTagProcessingTime);
                return directCommand(sendwCRC, sendwTRCAL, txBitCount, txBits, estimatedRxCount, additionalInfo.ToArray(), out headerBit, out rxBytes);
            }

            /// <summary>
            /// Sends a direct command to the tag
            /// </summary>
            /// <param name="sendwCRC">Specifies if the data should be sent with CRC</param>
            /// <param name="txBitCount">The count of bits to be sent</param>
            /// <param name="txBits">The bits to be sent</param>
            /// <param name="headerBit">Flag to signalize, if header bit was received</param>
            /// <param name="estimatedRxCount">The estimated count of rx bits</param>
            /// <param name="additionalInfo">Additional Info like estimated RX count or tag processing time</param>
            /// <param name="rxBytes">The bytes received from the tag</param>
            /// <returns>Success of the operation</returns>
            private bool directCommand(byte sendwCRC, byte sendwTRCAL, ushort txBitCount, byte[] txBits, ushort estimatedRxCount, byte[] additionalInfo, out bool headerBit, out byte[] rxBytes)
            {
                rxBytes = new byte[0];
                headerBit = true;

                byte txByteCount = (byte)(txBitCount >> 3);
                if ((txBitCount % 8) > 0)
                    txByteCount++;

                if (txByteCount != txBits.Length)
                {
                    Global.trc(3, "Direct Command - NOK - Data");
                    return false;
                }


                byte[] payload;
                List<byte> payloadList = new List<byte>();
                payloadList.Add((byte)sendwCRC);
                payloadList.Add((byte)sendwTRCAL);
                payloadList.Add((byte)(txBitCount >> 8));
                payloadList.Add((byte)txBitCount);
                payloadList.AddRange(txBits);
                payloadList.Add((byte)(estimatedRxCount >> 8));
                payloadList.Add((byte)estimatedRxCount);
                payloadList.AddRange(additionalInfo);

                payload = payloadList.ToArray();

                byte[] resultData;

                if (!customTagCommand((byte)0x09, payload, out resultData))
                    return false;

                headerBit = resultData[0] != 0;

                rxBytes = new byte[resultData.Length - 2];
                Array.Copy(resultData, 2, rxBytes, 0, resultData.Length - 2);

                return true;
            }

        #endregion

        /// <summary>
        /// Delegate is called everytime an inventory round ended.
        /// </summary>
        public delegate void InventoryRoundEndedHandler();

        /// <summary>
        /// Event is emitted everytime an inventory round ended.
        /// </summary>
        public event InventoryRoundEndedHandler InventoryRoundEnded;

        /// <summary>
        /// Parses the received notifcation data
        /// </summary>
        /// <param name="payload">Payload of the notification</param>
        protected override void notificationISR(byte[] payload)
        {
            if(payload.Count() < 1){
                base.notificationISR(payload);
                return;
            }

            switch (payload[0])
            {
                case 0x02:
                    if (InventoryRoundEnded != null)
                        InventoryRoundEnded();
                    break;

                default:
                    base.notificationISR(payload);
                    break;
            }

        }
    }
}
