/*
 * Copyright (c) 2008-2014, RF-Embedded GmbH
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.rfe.protocol;

import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicBoolean;

import com.rfe.JrfeGlobal;
import com.rfe.device.IProtocolDeviceInterface;
import com.rfe.helper.JrfeHelper;

/**
 * Implementation of the PUR specific functions
 */
public class JrfePURprotocolHandler extends JrfeProtocolHandler {

	/**
	 * Constructs a new object of the PUR protocol handler
	 * @param device	The communication device to the reader
	 */
	public JrfePURprotocolHandler(IProtocolDeviceInterface device) {
		super(device);
	}

	
	/**
	 * Tries to execute a block-write command
	 * @param epc			The known EPC of the tag
	 * @param mem_bank		The memory bank
	 * @param address		The address 
	 * @param passwd		The password
	 * @param blockSize		The block size of BlockWrite that should be used in words
	 * @param data			The data to write
	 * @throws JrfeProtocolException
	 */
    public void blockWriteToTag(byte[] epc, byte mem_bank, Short address, byte[] passwd, byte blockSize, byte[] data) throws JrfeProtocolException
    {
        if (data.length > 220)
        {
            JrfeGlobal.trc(3, "BlockWrite To Tag - NOK - Data");
            throw JrfeProtocolException.DataInputError("Data");
        }

        if (passwd.length != 4)
        {
            JrfeGlobal.trc(3, "BlockWrite To Tag - NOK - Data");
            throw JrfeProtocolException.DataInputError("Password");
        }

        byte[] payload;
        ArrayList<Byte> payloadList = new ArrayList<Byte>();
        payloadList.add((byte)epc.length);
        JrfeHelper.addRange(epc, payloadList);

        payloadList.add((byte)mem_bank);
        payloadList.add((byte)(address >> 8));
        payloadList.add((byte)(address >> 0));
        JrfeHelper.addRange(passwd, payloadList);
        payloadList.add((byte)blockSize);
        payloadList.add((byte)data.length);
        JrfeHelper.addRange(data, payloadList);

        payload = JrfeHelper.toByteArray(payloadList);

        customTagCommand((byte)0x03, payload);

        return;
    }

    /**
     * Retrieves the handle of a tag
     * @param epc		The known EPC of the tag
     * @return			The retrieved handle
     * @throws JrfeProtocolException
     */
    public byte[] getTagHandle(byte[] epc) throws JrfeProtocolException
    {
        byte[] handle = null;

        byte[] payload;
        ArrayList<Byte> payloadList = new ArrayList<Byte>();
        payloadList.add((byte)epc.length);
        JrfeHelper.addRange(epc, payloadList);
        payload = JrfeHelper.toByteArray(payloadList);

        byte[] resultData;

        resultData = customTagCommand((byte)0x04, payload);

        if (resultData.length < 2)
        	throw JrfeProtocolException.DataError(_LastSentMessage, payload);

        handle = new byte[2];
        System.arraycopy(resultData, 0, handle, 0, 2);

        return handle;
    }

    /**
     * Reads data direct from the handle of a tag
     * @param handle		The handle of the tag
     * @param mem_bank		The mem bank to read from
     * @param address		The mem address to read from
     * @param passwd		The passwd to be used
     * @param count			The count of data to read from
     * @return				The read data
     * @throws JrfeProtocolException
     */
    public byte[] readFromHandle(byte[] handle, byte mem_bank, Short address, byte[] passwd, byte count) throws JrfeProtocolException
    {
        byte[] data = null;

        if (handle.length != 2)
        {
            JrfeGlobal.trc(3, "Read From Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Handle");
        }

        if (passwd.length != 4)
        {
            JrfeGlobal.trc(3, "Read From Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Password");
        }

        byte[] payload;
        ArrayList<Byte> payloadList = new ArrayList<Byte>();
        JrfeHelper.addRange(handle, payloadList);

        payloadList.add((byte)mem_bank);
        payloadList.add((byte)(address >> 8));
        payloadList.add((byte)(address >> 0));
        JrfeHelper.addRange(passwd, payloadList);

        payloadList.add(count);
        payload = JrfeHelper.toByteArray(payloadList);

        byte[] resultData;

        resultData = customTagCommand((byte)0x05, payload);

        data = new byte[resultData.length - 1];
        System.arraycopy(resultData, 1, data, 0, resultData.length - 1);

        return data;
    }

    /**
     * Writes data direct to the handle of a tag
     * @param handle		The handle of the tag
     * @param mem_bank		The mem bank to write to
     * @param address		The mem address to write to
     * @param passwd		The passwd to be used
     * @param data			The data to be written
     * @throws JrfeProtocolException
     */
    public void writeToHandle(byte[] handle, byte mem_bank, Short address, byte[] passwd, byte[] data) throws JrfeProtocolException
    {
        if (handle.length != 2)
        {
            JrfeGlobal.trc(3, "Write To Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Handle");
        }
        
        if (data.length > 220)
        {
            JrfeGlobal.trc(3, "Write To Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Data");
        }

        if (passwd.length != 4)
        {
            JrfeGlobal.trc(3, "Write To Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Password");
        }

        byte[] payload;
        ArrayList<Byte> payloadList = new ArrayList<Byte>();
        JrfeHelper.addRange(handle, payloadList);
        payloadList.add((byte)mem_bank);
        payloadList.add((byte)(address >> 8));
        payloadList.add((byte)(address >> 0));
        JrfeHelper.addRange(passwd, payloadList);
        payloadList.add((byte)data.length);
        JrfeHelper.addRange(data, payloadList);
        payload = JrfeHelper.toByteArray(payloadList);

        customTagCommand((byte)0x06, payload);

        return;
    }
    
    /**
     * BlockWrites data direct to the handle of a tag
     * @param handle		The handle of the tag
     * @param mem_bank		The mem bank to write to
     * @param address		The mem address to write to
     * @param passwd		The passwd to be used
     * @param blockSize		The block size of BlockWrite that should be used in words
     * @param data			The data to be written
     * @throws JrfeProtocolException
     */
    public void blockWriteToHandle(byte[] handle, byte mem_bank, Short address, byte[] passwd, byte blockSize, byte[] data) throws JrfeProtocolException
    {
        if (handle.length != 2)
        {
            JrfeGlobal.trc(3, "BlockWrite To Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Handle");
        }

        if (data.length > 220)
        {
            JrfeGlobal.trc(3, "BlockWrite To Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Data");
        }

        if (passwd.length != 4)
        {
            JrfeGlobal.trc(3, "BlockWrite To Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Password");
        }

        byte[] payload;
        ArrayList<Byte> payloadList = new ArrayList<Byte>();
        JrfeHelper.addRange(handle, payloadList);
        payloadList.add((byte)mem_bank);
        payloadList.add((byte)(address >> 8));
        payloadList.add((byte)(address >> 0));
        JrfeHelper.addRange(passwd, payloadList);
        payloadList.add((byte)blockSize);
        payloadList.add((byte)data.length);
        JrfeHelper.addRange(data, payloadList);
        payload = JrfeHelper.toByteArray(payloadList);

        customTagCommand((byte)0x08, payload);

        return;
    }

    /**
     * Sends a custom tag command to the tag
     * @param handle			The handle of the tag
     * @param command			The two byte command to be sent
     * @param passwd			The passwd to be used
     * @param txBitCount		The count of bits to be sent
     * @param txBits			The bits to be sent
     * @param estimatedRxCount	The estimated count of data to receive
     * @param headerBit			Flag to signalize, if header bit was received
     * @return					The bytes received from the tag
     * @throws JrfeProtocolException
     */
    public byte[] customGen2Command(byte[] handle, byte[] command, byte[] passwd, byte txBitCount, byte[] txBits, byte estimatedRxCount, AtomicBoolean headerBit) throws JrfeProtocolException
    {
    	byte[] rxBytes = null;
        headerBit.set(true);

        if (handle.length != 2)
        {
            JrfeGlobal.trc(3, "Custom Tag Command Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Handle");
        }

        if (command.length != 2)
        {
            JrfeGlobal.trc(3, "Custom Tag Command Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Command");
        }

        if (passwd.length != 4)
        {
            JrfeGlobal.trc(3, "Custom Tag Command Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("Password");
        }

        byte txByteCount = (byte)(txBitCount >> 3);
        if ((txBitCount % 8) > 0)
            txByteCount++;

        if (txByteCount != txBits.length)
        {
            JrfeGlobal.trc(3, "Custom Tag Command Handle - NOK - Data");
            throw JrfeProtocolException.DataInputError("TxBits");
        }


        byte[] payload;
        ArrayList<Byte> payloadList = new ArrayList<Byte>();
        JrfeHelper.addRange(handle, payloadList);
        JrfeHelper.addRange(command, payloadList);
        JrfeHelper.addRange(passwd, payloadList);
        payloadList.add((byte)txBitCount);
        JrfeHelper.addRange(txBits, payloadList);
        payloadList.add((byte)estimatedRxCount);

        payload = JrfeHelper.toByteArray(payloadList);

        byte[] resultData;

        resultData = customTagCommand((byte)0x07, payload);

        headerBit.set(resultData[0] != 0);

        rxBytes = new byte[resultData.length - 2];
        System.arraycopy(resultData, 2, rxBytes, 0, resultData.length - 2);

        return rxBytes;
    }

}
