/*
 * Copyright (c) 2008-2014, RF-Embedded GmbH
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.rfe.protocol;

import com.rfe.helper.JrfeHelper;

/**
 * Class to store a tag event
 */
public class JrfeTagEvent {
	
	/** The tag ID */
    public byte[] tagId;

    /** The antenna infos */
    public boolean hasAntenna;
    public byte antennaId;

    /** The rssi infos */
    public boolean hasRSSI;
    public byte[] rssi;

    /** The frequency infos */
    public boolean hasReadFrequency;
    public long readFrequency;

    /** The memory infos */
    public boolean hasMemory;
    public byte memBank;
    public short memAddr;
    public byte[] memData;

    /** The trigger info */
    public boolean hasTrigger;
    public byte trigger;

    /** The handle info */
    public boolean hasHandle;
    public byte[] handle;

    /** The state info */
    public boolean hasState;
    public short state;

    /** The battery info */
    public boolean hasBattery;
    public byte battery;

    /** The PC info */
    public boolean hasPC;
    public byte[] pc;

    /** The message ID */
    public boolean hasMessageId;
    public byte messageId;

    /** The application info */
    public boolean hasApplicationInfo;
    public byte[] applicationInfo;

    /**
     * Constructs a new empty tag event instance
     */
    public JrfeTagEvent()
    {
        tagId = new byte[0];
        hasAntenna = false;
        hasRSSI = false;
        hasReadFrequency = false;
        hasMemory = false;
        hasTrigger = false;
        hasHandle = false;
        hasState = false;
        hasBattery = false;
        hasPC = false;
        hasMessageId = false;
        hasApplicationInfo = false;
    }

    /**
     * Copy constructos for a tag event
     * @param other		The other tag event
     */
    public JrfeTagEvent(JrfeTagEvent other)
    {
        tagId = new byte[other.tagId.length];
        System.arraycopy(other.tagId, 0, tagId, 0, tagId.length);

        hasAntenna = other.hasAntenna;
        if (hasAntenna)
        {
            antennaId = other.antennaId;
        }

        hasRSSI = other.hasRSSI;
        if (hasRSSI)
        {
            rssi = new byte[other.rssi.length];
            System.arraycopy(other.rssi, 0, rssi, 0, rssi.length);
        }

        hasReadFrequency = other.hasReadFrequency;
        if (hasReadFrequency)
        {
            readFrequency = other.readFrequency;
        }

        hasMemory = other.hasMemory;
        if (hasMemory)
        {
            memBank = other.memBank;
            memAddr = other.memAddr;
            memData = new byte[other.memData.length];
            System.arraycopy(other.memData, 0, memData, 0, memData.length);
        }

        hasTrigger = other.hasTrigger;
        if (hasTrigger)
        {
            trigger = other.trigger;
        }

        hasHandle = other.hasHandle;
        if (hasHandle)
        {
            handle = new byte[other.handle.length];
            System.arraycopy(other.handle, 0, handle, 0, handle.length);
        }

        hasState = other.hasState;
        if (hasState)
        {
            state = other.state;
        }

        hasBattery = other.hasBattery;
        if (hasBattery)
        {
            battery = other.battery;
        }

        hasPC = other.hasPC;
        if (hasBattery)
        {
            pc = new byte[other.pc.length];
            System.arraycopy(other.pc, 0, pc, 0, pc.length);
        }

        hasMessageId = other.hasMessageId;
        if (hasMessageId)
        {
            messageId = other.messageId;
        }

        hasApplicationInfo = other.hasApplicationInfo;
        if (hasApplicationInfo)
        {
            applicationInfo = new byte[other.applicationInfo.length];
            System.arraycopy(other.applicationInfo, 0, applicationInfo, 0, applicationInfo.length);
        }
    }

    /**
     * Converts a tag event to a string
     */
    public String toString()
    {
        StringBuilder ret = new StringBuilder();

        ret.append(JrfeHelper.toHexString(tagId));

        if (hasAntenna)
        {
            ret.append(", Antenna #{1}" + antennaId);
        }

        if (hasRSSI)
        {
            ret.append(", RSSI " + JrfeHelper.toHexString(rssi));
        }

        if (hasReadFrequency)
        {
            ret.append(", Frequency " + readFrequency);
        }

        if (hasMemory)
        {
            ret.append(", Data " + memBank + "@" + memAddr + JrfeHelper.toHexString(memData));
        }

        if (hasTrigger)
        {
            ret.append(", Trigger " + trigger);
        }

        if (hasHandle)
        {
            ret.append(", Handle " + JrfeHelper.toHexString(handle));
        }

        if (hasState)
        {
            ret.append(", State " + state);
        }

        if (hasBattery)
        {
            ret.append(", Battery " + battery);
        }

        if (hasPC)
        {
            ret.append(", PC " + JrfeHelper.toHexString(pc));
        }

        if (hasMessageId)
        {
            ret.append(", MessageID " + messageId);
        }

        if (hasApplicationInfo)
        {
            ret.append(", ApplicationInfo " + JrfeHelper.toHexString(applicationInfo));
        }

        return ret.toString();
    }
}
