/*
 * Copyright (c) 2008-2014, RF-Embedded GmbH
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.rfe.test.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketAddress;
import com.rfe.device.IProtocolDeviceInterface;

/**
 * Implementation of a TCP device
 */
public class TCPDevice extends IProtocolDeviceInterface {
	
	/** The socket instance */
	private Socket m_socket = new Socket();
	/** The target socket address */
	private SocketAddress m_socketAddress = null;
	
	/** The data input stream */
	private DataInputStream m_inputStream;
	/** The data output stream */
	private DataOutputStream m_outputStream;
	
	/** The WorkerThread instance */
	private WorkerThread m_workerThread = new WorkerThread();
	
	/**
	 * Creates a new tcp socket to the given ip and the given port
	 * @param ip	Target ip
	 * @param port	Target port
	 */
	public TCPDevice (String ip, int port)
	{
		m_socketAddress = new InetSocketAddress(ip, port);
	}

	@Override
	public boolean open() {
		if(m_socketAddress == null)
			return false;
		if(m_socket.isConnected())
			return true;
		try {
			m_socket.connect(m_socketAddress);
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		}
		
		if(!m_socket.isConnected())
			return false;
		
		try {
			m_inputStream = new DataInputStream(m_socket.getInputStream());
			m_outputStream = new DataOutputStream(m_socket.getOutputStream());
		} catch (IOException e) {
			e.printStackTrace();
			try {
				m_socket.close();
			} catch (IOException e1) {
				e1.printStackTrace();
			}
			return false;
		}
		
		new Thread(m_workerThread).start();

		return true;
	}

	@Override
	public boolean close() {
		
		m_workerThread.stop();
		
		if(!m_socket.isConnected())
			return true;
		try {
			m_socket.close();
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	@Override
	public boolean send(byte[] data) {
		if(!m_socket.isConnected())
			return false;
		try {
			m_outputStream.write(data);
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	/**
	 * Enum to hold the state of the worker thread
	 */
    private enum WorkerThreadState {
        STOPPED,
        RUNNING,
        STOPPING
    }

    /**
     * WorkerThread to read from the TCP socket
     */
	private class WorkerThread implements Runnable
	{
		/**
		 * The state of the WorkerThread
		 */
	    private WorkerThreadState m_state = WorkerThreadState.STOPPED;
	    
	    /**
	     * The polling timeout in milliseconds
	     */
	    private static final int POLL_TIMEOUT = 1;

	    /**
	     * The buffer size of the read buffer
	     */
	    private static final int BUFSIZE = 4096;

	    /**
	     * Returns the state of the WorkerThread
	     * @return
	     */
	    private synchronized WorkerThreadState getState() 
	    {
	        return m_state;
	    }
	    
	    /**
	     * Sets the state of the WorkerThread
	     * @return
	     */
	    private synchronized void setState(WorkerThreadState state) 
	    {
	        m_state = state;
	    }
	    
	    /**
	     * The run method of the WorkerThread
	     */
		public void run()
		{
			setState(WorkerThreadState.RUNNING);
			
			byte[] readData = new byte[BUFSIZE];
			
            Listener listener = getListener();
            if (listener != null) {
                listener.onConnectionOpened(true);
            }
			
			while(getState() == WorkerThreadState.RUNNING)
			{
				if(m_socket.isConnected())
				{
			        int len = 0;
			        try {
			        	len = m_inputStream.available();
			        	if(len > BUFSIZE)
			        		len = BUFSIZE;
			        	if(len > 0)
			        	{
			        		m_inputStream.read(readData, 0, len);
			        	}
					} catch (IOException e) {
						e.printStackTrace();
					}
			        
			        if (len > 0) {
			            listener = getListener();
			            if (listener != null) {
			                final byte[] data = new byte[len];
			                System.arraycopy(readData, 0, data, 0, len);
			                listener.onNewData(data);
			            }
			        }
			        
			        try {
						Thread.sleep(POLL_TIMEOUT);
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}
				else
				{
					stop();
		            listener = getListener();
		            if (listener != null) {
		            	listener.onConnectionClosed();
		            }					
				}
			}
			
			setState(WorkerThreadState.STOPPED);
		}
		
		/**
		 * Stops the WorkerThread
		 */
		public void stop()
		{
	        if (getState() == WorkerThreadState.RUNNING) 
	        {
				setState(WorkerThreadState.STOPPING);
	        }
		}
	}
	


}
